/******************************************************************************
 * filename: callbacks.cpp                                                    *
 *                                                                            *
 * author:   Christopher Oat                                                  *
 *           ATI Research, Inc.                                               *
 *           3D Application Research Group                                    *
 *           email: coat@ati.com                                              *
 *                                                                            *
 * description: routines to handle GLUT callback events.                      *
 *                                                                            *
 *                                                                            *
 ******************************************************************************
 $Header::                                                                    $
 ******************************************************************************
 *   (C) 2001 ATI Research, Inc.  All rights reserved.                        *
 ******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <GL/glut.h>

#include "globals.h"
#include "callbacks.h"
#include "dot3.h"
#include "light.h"
#include "matrix.h"

#ifndef PI
#define PI 3.1415926535
#endif

static int mouseLeftDown=0;
static int mouseLeftX = 0;
static int mouseLeftY = 0;

static int mouseRightDown=0;
static int mouseRightX = 0;
static int mouseRightY = 0;

#define WIREFRAME_FLAG 0x1
#define AUTOPILOT_FLAG 0x2
#define DRAWVECTR_FLAG 0x4

static unsigned char callbackFlags = AUTOPILOT_FLAG;
static int callbacksTex=0;
static int callbacksMode=DOT3_DOT3BUMP;
static GLfloat callbacksZoom=-200.0;

GLfloat lightMat[16] = {1.0, 0.0, 0.0, 0.0,
                        0.0, 1.0, 0.0, 0.0,
                        0.0, 0.0, 1.0, 0.0,
                        0.0, 0.0, 0.0, 1.0};

// setup glut handler routines
void callbacksInit (void)
{
  glutDisplayFunc(callbacksDisplay);
  glutReshapeFunc(callbacksReshape); 
  glutSpecialFunc(callbacksSpecial);
  glutKeyboardFunc(callbacksKeyboard);
  glutMouseFunc(callbacksMouse);
  glutMotionFunc(callbacksMotion);

  printf("Radeon Simple Dot3 Bump Demo: May 15, 2001\n");
  printf(" t             - Cycle through textures\n");
  printf(" w             - Toggle wireframe\n");
  printf(" b             - Show base map only\n");
  printf(" l             - Show L map only\n");
  printf(" n             - Show N map only\n");
  printf(" d             - Show DOT3 Bump mapping <default>\n");
  printf(" UP/DOWN ARROW - Zoom In/Out\n");
  printf(" LEFT + MOUSE  - Rotate Object\n");
  printf(" RIGHT + MOUSE - Rotate Light\n");
  printf(" SPACE         - Toggle Auto Motion\n");
  printf(" ESC           - Quit program\n");
  printf("\n");

  return;
}


// handle window reshape events
void callbacksReshape(int width, int height)
{
  screenWidth = width;
  screenHeight = height;

  glViewport(0, 0, width, height);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(fov,((float)screenWidth)/((float)screenHeight),0.01f, 5000.0f);//clipNear,clipFar);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glLoadMatrixf(viewMatrix);
 

  glutPostRedisplay();

}


// handle display events
void callbacksDisplay(void)
{
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  glShadeModel(GL_SMOOTH);

  if ((callbackFlags & WIREFRAME_FLAG) == WIREFRAME_FLAG)
  {
     glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  }
  else
  {
     glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  }

  glMatrixMode(GL_MODELVIEW);

  glPushMatrix();
  glLoadIdentity();
  glTranslatef(0.0, 0.0, callbacksZoom);
  glPushMatrix();
  glMultMatrixf(viewMatrix);
 
  if ((callbackFlags & WIREFRAME_FLAG) == WIREFRAME_FLAG)
  {
     dot3Display(callbacksTex, DOT3_WIREFRAME);
  }
  else
  {
     dot3Display(callbacksTex, callbacksMode);
  }


  if ((callbackFlags & DRAWVECTR_FLAG) == DRAWVECTR_FLAG)
     dot3DisplayVectors();

  glPopMatrix();

  if ((callbackFlags & AUTOPILOT_FLAG) == AUTOPILOT_FLAG)
  {
     glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
         glLoadIdentity();
         glMultMatrixf(lightMat);
         glRotatef(0.25, 0.0f, 0.0f, 1.0f);
         glGetFloatv(GL_MODELVIEW_MATRIX, lightMat);
      glPopMatrix();
     
  }

  glMultMatrixf(lightMat);
  if ((callbackFlags & WIREFRAME_FLAG) == WIREFRAME_FLAG)
  {
     lightDisplay(GL_TRUE);
  }
  else
  {
     lightDisplay(GL_FALSE);
  }
  

  glPopMatrix();
  
  
  
  glutSwapBuffers();
  glutPostRedisplay();
  return;
}


void callbacksSpecial(int key, int x, int y) 
{
  callbacksKeyboard(key,x,y);
}


// keyboard event handler
void callbacksKeyboard(unsigned char key, int x, int y)
{

  switch (key) 
    {
 
      case GLUT_KEY_UP   : callbacksZoom += 10.0f;
                           if (callbacksZoom >= clipNear)
                              callbacksZoom = clipNear - 10.0f;
                           break;

      case GLUT_KEY_DOWN : callbacksZoom -= 10.0;
                           if (callbacksZoom <= (-clipFar)+140.0f)
                              callbacksZoom = (-clipFar) + 140.0f;
                           break;

      case 'n'           : 
      case 'N'           : callbacksMode = DOT3_N_MAP_ONLY;
                           break;      

      case 'l'           : 
      case 'L'           : callbacksMode = DOT3_L_MAP_ONLY;
                           break;      
      case 't'           : 
      case 'T'           : callbacksTex += 1;
                           callbacksTex = ((callbacksTex >= dot3GetNumTex())? 0 : callbacksTex);
                           break;      
      case 'v'           : 
      case 'V'           : callbackFlags ^= DRAWVECTR_FLAG;
                           break;      
      case 'b'           : 
      case 'B'           : callbacksMode = DOT3_BASE_MAP_ONLY;
                           break;      

      case 'd'           : 
      case 'D'           : callbacksMode = DOT3_DOT3BUMP;
                           break;      

      case 'w'           : 
      case 'W'           : callbackFlags ^= WIREFRAME_FLAG;
                           break;      

      case ' '           : callbackFlags ^= AUTOPILOT_FLAG;
                           break;      

      case 'q'           :
      case 'Q'           :
      case 27            : exit(0);
                           break;

  }

  glutPostRedisplay();
  return;
}

// handle mouse events 
void callbacksMouse (int button, int state, int x, int y)
{
   

  // change camera angle
  if (button == GLUT_LEFT_BUTTON)
    {
      switch (state)
        {
          
        case GLUT_DOWN : mouseLeftDown=1;
                         mouseLeftX=x;
                         mouseLeftY=y;
                         break;                         
        case GLUT_UP   : mouseLeftDown=0;
                         break;
        }
    }

  // move swarm
  if (button == GLUT_RIGHT_BUTTON)
    {
      switch (state)
        {
                         
        case GLUT_DOWN : mouseRightDown=1;
                         mouseRightX=x;
                         mouseRightY=y;
                         break;
        case GLUT_UP   : mouseRightDown=0;
                         break;
        }
    }

  
  glutPostRedisplay();
  return;
}


// mouse move events within the window
void callbacksMotion (int x, int y)
{
   
    // changes modelview matrix
   if (mouseLeftDown && (!mouseRightDown))
   {
      glMatrixMode(GL_MODELVIEW);
      GLfloat m[16];
      matrixInvert(viewMatrix, m);
      GLfloat v[3] = {(y - mouseLeftY) * 0.3f, (x - mouseLeftX) * 0.3f, 0.0f };
      GLfloat o[3];
      vecMat3x3Mult(v, m, o);
      glPushMatrix();
         glLoadIdentity();
         glMultMatrixf(viewMatrix);

         glRotatef(o[0], 1.0f, 0.0f, 0.0f);
         glRotatef(o[1], 0.0f, 1.0f, 0.0f);
         glRotatef(o[2], 0.0f, 0.0f, 1.0f);
         glGetFloatv(GL_MODELVIEW_MATRIX, viewMatrix);
      glPopMatrix();
      mouseLeftX = x;
      mouseLeftY = y;
   }


   // moves light source
   if (mouseRightDown && (!mouseLeftDown))
   {
      glMatrixMode(GL_MODELVIEW);
      GLfloat m[16];
      matrixInvert(lightMat, m);
      GLfloat v[3] = {(y - mouseRightY) * 0.3f, (x - mouseRightX) * 0.3f, 0.0f };
      GLfloat o[3];
      vecMat3x3Mult(v, m, o);
      glPushMatrix();
         glLoadIdentity();
         glMultMatrixf(lightMat);

         glRotatef(o[0], 1.0f, 0.0f, 0.0f);
         glRotatef(o[1], 0.0f, 1.0f, 0.0f);
         glRotatef(o[2], 0.0f, 0.0f, 1.0f);
         glGetFloatv(GL_MODELVIEW_MATRIX, lightMat);
      glPopMatrix();
      mouseRightX = x;
      mouseRightY = y;
   }
   
  
   glutPostRedisplay();
   return;
}



