/******************************************************************************
 * filename: dot3.cpp                                                         *
 *                                                                            *
 * author:   Christopher Oat                                                  *
 *           ATI Research, Inc.                                               *
 *           3D Application Research Group                                    *
 *           email: coat@ati.com                                              *
 *                                                                            *
 * description: loads textures and displays a textured quad with per pixel    *
 *              dot3 bump mapping.                                            *
 *                                                                            *
 ******************************************************************************
 $Header::                                                                    $
 ******************************************************************************
 *   (C) 2001 ATI Research, Inc.  All rights reserved.                        *
 ******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <GL/gl.h>
#include <GL/glext.h>
#include <GL/glut.h>

#include "globals.h"
#include "GLTexture.h"
#include "GLErr.h"
#include "dot3.h"
#include "light.h"
#include "matrix.h"

#define DOT3_NUM_MAPS 5
#define DOT3_DRAW_VEC_SIZE 15

static int dot3IsInitialized=0;

static GLuint dot3TextureID[DOT3_NUM_MAPS][2];

static char* dot3MapFileName[][2] = {{FILE_PATH "texture0.tga", FILE_PATH "texture0DOT3.tga"},
                                     {FILE_PATH "texture1.tga", FILE_PATH "texture1DOT3.tga"},
                                     {FILE_PATH "texture2.tga", FILE_PATH "texture2DOT3.tga"},
                                     {FILE_PATH "texture3.tga", FILE_PATH "texture3DOT3.tga"},
                                     {FILE_PATH "texture3.tga", FILE_PATH "texture4DOT3.tga"}};
                                     


static ATI_GLTexture *dot3TextureMap[DOT3_NUM_MAPS][2];

static GLfloat tangentMatrix[16] = {1.0, 0.0, 0.0, 0.0,
                                   0.0,-1.0, 0.0, 0.0,
                                   0.0, 0.0, 1.0, 0.0,
                                   0.0, 0.0, 0.0, 1.0};

// load texture maps and make texture objects.
void dot3Init (void)
{

   // Reference to global error message.
   extern char gszErrMsg[];
   gszErrMsg[0] = '\0';
   
   printf("Loading Texture Maps\n");
   printf("--------------------\n");

   for (int n=0; n < DOT3_NUM_MAPS; ++n)
   {
      // First load base map
      printf("%d: %s", n, dot3MapFileName[n][0]);
      gszErrMsg[0] = '\0';

      glGenTextures(2, dot3TextureID[n]);

      dot3TextureMap[n][0] = new ATI_GLTexture(dot3MapFileName[n][0], dot3TextureID[n][0], GL_TRUE, GL_RGB, GL_RGB);
      
      if (gszErrMsg[0] != '\0')
         printf("\tError loading %s: %s\n", dot3MapFileName[n][0], gszErrMsg);
      else
         printf("\tDone!\t");

      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER_ARB);
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER_ARB);
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
      

      // Second load the DOT3 bump map
      printf("%s", dot3MapFileName[n][1]);
      gszErrMsg[0] = '\0';

      dot3TextureMap[n][1] = new ATI_GLTexture(dot3MapFileName[n][1], dot3TextureID[n][1], GL_TRUE, GL_RGB, GL_RGB);
      
      if (gszErrMsg[0] != '\0')
         printf("\tError loading %s: %s\n", dot3MapFileName[n][1], gszErrMsg);
      else
         printf("\tDone!\n");

      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER_ARB);
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER_ARB);
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
      
   }

   printf("\n");


   dot3IsInitialized=1;

   return;
}


// display a lit quad with per pixel dot3 bump mapping
void dot3Display (int tex, int mode)
{
   tex = ((tex > DOT3_NUM_MAPS-1) ? DOT3_NUM_MAPS-1 : tex);

   if (tex < 0) tex = -tex;

   // dot3 bump mapping computes (N.L) per pixel.  we do this by storing the N
   // vector in a texture (known as the dot3 bump map) and then L is computed
   // per vertex and is stored at each vertex as a color.  This color is 
   // interpolated between vertices by OpenGL when the shade model is set to GL_SMOOTH.
   // N.L is computed by the texture combiner and then modulated with the base texture.
   // 
   // Note that L vector must be rotated into tangent space (aka texture space) so
   // that it is in the same space as N.

   if (dot3IsInitialized)
   {
      
      // reset texture unit state
      glActiveTexture(GL_TEXTURE0_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE1_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE2_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);


      // set up texture units for rendering N Map only
      if (mode == DOT3_N_MAP_ONLY)
      {
         glActiveTexture(GL_TEXTURE0_ARB);
         glBindTexture (GL_TEXTURE_2D, dot3TextureID[tex][1]);
         glEnable(GL_TEXTURE_2D);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_ARB);
         glTexEnvf(GL_TEXTURE_ENV, GL_COMBINE_RGB_ARB, GL_REPLACE);   
      
         glTexEnvf(GL_TEXTURE_ENV, GL_SOURCE0_RGB_ARB, GL_TEXTURE);
         glTexEnvf(GL_TEXTURE_ENV, GL_OPERAND0_RGB_ARB, GL_SRC_COLOR);
      }

      // set up texture units for rendering L Map only
      if (mode == DOT3_L_MAP_ONLY)
      {
         glActiveTexture(GL_TEXTURE0_ARB);
         glBindTexture (GL_TEXTURE_2D, dot3TextureID[tex][1]);
         glEnable(GL_TEXTURE_2D);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_ARB);
         glTexEnvf(GL_TEXTURE_ENV, GL_COMBINE_RGB_ARB, GL_REPLACE);   
      
         glTexEnvf(GL_TEXTURE_ENV, GL_SOURCE0_RGB_ARB, GL_PRIMARY_COLOR_ARB);
         glTexEnvf(GL_TEXTURE_ENV, GL_OPERAND0_RGB_ARB, GL_SRC_COLOR);
      }

      // set up texture units for rendering Base Map only
      if (mode == DOT3_BASE_MAP_ONLY)
      {
         glActiveTexture(GL_TEXTURE0_ARB);
         glBindTexture (GL_TEXTURE_2D, dot3TextureID[tex][0]);
         glEnable(GL_TEXTURE_2D);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_ARB);
         glTexEnvf(GL_TEXTURE_ENV, GL_COMBINE_RGB_ARB, GL_REPLACE);   
      
         glTexEnvf(GL_TEXTURE_ENV, GL_SOURCE0_RGB_ARB, GL_TEXTURE);
         glTexEnvf(GL_TEXTURE_ENV, GL_OPERAND0_RGB_ARB, GL_SRC_COLOR);
      }

      // set up texture units for rendering BaseMap*(N.L)
      if (mode == DOT3_DOT3BUMP)
      {
         // UNIT 0 
         // find dot product of N (stored in the texture map) and L (stored
         // as the PRIMARY_COLOR).
         glActiveTexture(GL_TEXTURE0_ARB);
         glBindTexture (GL_TEXTURE_2D, dot3TextureID[tex][1]);
         glEnable(GL_TEXTURE_2D);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_ARB);
         glTexEnvf(GL_TEXTURE_ENV, GL_COMBINE_RGB_ARB, GL_DOT3_RGB_ARB); 
      
         glTexEnvf(GL_TEXTURE_ENV, GL_SOURCE0_RGB_ARB, GL_TEXTURE);
         glTexEnvf(GL_TEXTURE_ENV, GL_OPERAND0_RGB_ARB, GL_SRC_COLOR);
         glTexEnvf(GL_TEXTURE_ENV, GL_SOURCE1_RGB_ARB, GL_PRIMARY_COLOR_ARB);
         glTexEnvf(GL_TEXTURE_ENV, GL_OPERAND1_RGB_ARB, GL_SRC_COLOR);
      
         // UNIT 1
         // modulate the base texture by N.L
         
         glActiveTexture(GL_TEXTURE1_ARB);
         glBindTexture (GL_TEXTURE_2D, dot3TextureID[tex][0]);
         glEnable(GL_TEXTURE_2D);
         
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_ARB);
         glTexEnvf(GL_TEXTURE_ENV, GL_COMBINE_RGB_ARB, GL_MODULATE);

         glTexEnvf(GL_TEXTURE_ENV, GL_SOURCE0_RGB_ARB, GL_PREVIOUS_ARB);
         glTexEnvf(GL_TEXTURE_ENV, GL_OPERAND0_RGB_ARB, GL_SRC_COLOR);
         glTexEnvf(GL_TEXTURE_ENV, GL_SOURCE1_RGB_ARB, GL_TEXTURE);
         glTexEnvf(GL_TEXTURE_ENV, GL_OPERAND1_RGB_ARB, GL_SRC_COLOR);

      }

      static GLfloat lpES[3];  // light position in Eye space
      static GLfloat lpOS[3];  // light position in Object space
      static GLfloat lv[3];    // vector from vertex to light
      static GLfloat lv_ts[3];    // light vector in tangent space
      
      static GLfloat modelViewMatrix[16];
      static GLfloat modelViewMatrixInverse[16];
      
      glGetFloatv(GL_MODELVIEW_MATRIX, modelViewMatrix);
      matrixInvert(modelViewMatrix, modelViewMatrixInverse);
      
      // get light's position in eye space
      lightGetPos(lpES);

      // convert from eye space to object space
      vecMatMult(lpES, modelViewMatrixInverse, lpOS);


      glBegin(GL_TRIANGLE_STRIP);

      // compute L and store as vertex color
      lv[0] = (lpOS[0] + 50.0); 
      lv[1] = (lpOS[1] + 50.0); 
      lv[2] = (lpOS[2] - 0.0); 
      vecNormalize(lv);

      // rotate into tangent space
      vecMat3x3Mult(lv, tangentMatrix, lv_ts);

      //scale and bias
      lv_ts[0] = lv_ts[0] * 0.5 + 0.5; 
      lv_ts[1] = lv_ts[1] * 0.5 + 0.5; 
      lv_ts[2] = lv_ts[2] * 0.5 + 0.5; 

      glMultiTexCoord2f(GL_TEXTURE0_ARB, 0.0, 0.0);
      glMultiTexCoord2f(GL_TEXTURE1_ARB, 0.0, 0.0);
      glColor3fv(lv_ts);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(-50.0, -50.0, 0.0);
      




      // compute L and store as vertex color
      lv[0] = (lpOS[0] - 50.0); 
      lv[1] = (lpOS[1] + 50.0); 
      lv[2] = (lpOS[2] - 0.0); 
      vecNormalize(lv);

      //contert to tangent space
      vecMat3x3Mult(lv, tangentMatrix, lv_ts);

      //scale and bias
      lv_ts[0] = lv_ts[0] * 0.5 + 0.5; 
      lv_ts[1] = lv_ts[1] * 0.5 + 0.5; 
      lv_ts[2] = lv_ts[2] * 0.5 + 0.5; 

      glMultiTexCoord2f(GL_TEXTURE0_ARB, 1.0, 0.0);
      glMultiTexCoord2f(GL_TEXTURE1_ARB, 1.0, 0.0);
      glColor3fv(lv_ts);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(50.0, -50.0, 0.0);




      // compute L and store as vertex color
      lv[0] = (lpOS[0] + 50.0); 
      lv[1] = (lpOS[1] - 50.0); 
      lv[2] = (lpOS[2] - 0.0); 
      vecNormalize(lv);
      
      //contert to tangent space
      vecMat3x3Mult(lv, tangentMatrix, lv_ts);

      //scale and bias
      lv_ts[0] = lv_ts[0] * 0.5 + 0.5; 
      lv_ts[1] = lv_ts[1] * 0.5 + 0.5; 
      lv_ts[2] = lv_ts[2] * 0.5 + 0.5; 

      glMultiTexCoord2f(GL_TEXTURE0_ARB, 0.0, 1.0);
      glMultiTexCoord2f(GL_TEXTURE1_ARB, 0.0, 1.0);
      glColor3fv(lv_ts);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(-50.0, 50.0, 0.0);




      // compute L and store as vertex color
      lv[0] = (lpOS[0] - 50.0); 
      lv[1] = (lpOS[1] - 50.0); 
      lv[2] = (lpOS[2] - 0.0); 
      vecNormalize(lv);

      //contert to tangent space
      vecMat3x3Mult(lv, tangentMatrix, lv_ts);

      //scale and bias
      lv_ts[0] = lv_ts[0] * 0.5 + 0.5; 
      lv_ts[1] = lv_ts[1] * 0.5 + 0.5; 
      lv_ts[2] = lv_ts[2] * 0.5 + 0.5; 

      glMultiTexCoord2f(GL_TEXTURE0_ARB, 1.0, 1.0);
      glMultiTexCoord2f(GL_TEXTURE1_ARB, 1.0, 1.0);
      glColor3fv(lv_ts);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(50.0, 50.0, 0.0);

      
      glEnd();

      
      // reset texture unit state
      glActiveTexture(GL_TEXTURE0_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE1_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

   }
   else
   {
      printf("Error: call to dot3Display() without prior dot3Init()\n");
   }

   return;
}

// display L vector per vertex
void dot3DisplayVectors (void)
{
   if (dot3IsInitialized)
   {
      // reset texture unit state
      glActiveTexture(GL_TEXTURE0_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE1_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE2_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

      static GLfloat lpES[3];
      static GLfloat lpOS[3];
      static GLfloat lv[3];
            
      static GLfloat modelViewMatrix[16];
      static GLfloat modelViewMatrixInverse[16];
      
      glGetFloatv(GL_MODELVIEW_MATRIX, modelViewMatrix);
      matrixInvert(modelViewMatrix, modelViewMatrixInverse);
      

      lightGetPos(lpES);
      vecMatMult(lpES, modelViewMatrixInverse, lpOS);
     
      glBegin(GL_LINES);

      glColor3f(0.0, 1.0, 1.0);

      lv[0] = (lpOS[0] + 50.0); 
      lv[1] = (lpOS[1] + 50.0); 
      lv[2] = (lpOS[2] - 0.0); 
      vecNormalize(lv);
      
      glVertex3f(-50.0, -50.0, 0.0);
      glVertex3f(-50.0+(lv[0]*DOT3_DRAW_VEC_SIZE),
                 -50.0+(lv[1]*DOT3_DRAW_VEC_SIZE),
                   0.0+(lv[2]*DOT3_DRAW_VEC_SIZE));
      

      lv[0] = (lpOS[0] - 50.0); 
      lv[1] = (lpOS[1] + 50.0); 
      lv[2] = (lpOS[2] - 0.0); 
      vecNormalize(lv);
      glVertex3f(50.0, -50.0, 0.0);
      glVertex3f( 50.0+(lv[0]*DOT3_DRAW_VEC_SIZE),
                 -50.0+(lv[1]*DOT3_DRAW_VEC_SIZE),
                   0.0+(lv[2]*DOT3_DRAW_VEC_SIZE));



      lv[0] = (lpOS[0] - 50.0); 
      lv[1] = (lpOS[1] - 50.0); 
      lv[2] = (lpOS[2] - 0.0); 
      vecNormalize(lv);
      glVertex3f(50.0, 50.0, 0.0);
      glVertex3f( 50.0+(lv[0]*DOT3_DRAW_VEC_SIZE),
                  50.0+(lv[1]*DOT3_DRAW_VEC_SIZE),
                   0.0+(lv[2]*DOT3_DRAW_VEC_SIZE));


      lv[0] = (lpOS[0] + 50.0); 
      lv[1] = (lpOS[1] - 50.0); 
      lv[2] = (lpOS[2] - 0.0); 
      vecNormalize(lv);
      glVertex3f(-50.0, 50.0, 0.0);
      glVertex3f(-50.0+(lv[0]*DOT3_DRAW_VEC_SIZE),
                  50.0+(lv[1]*DOT3_DRAW_VEC_SIZE),
                   0.0+(lv[2]*DOT3_DRAW_VEC_SIZE));
      
      glEnd();

   }
   else
   {
      printf("Error: call to dot3DisplayVectors() without prior dot3Init()\n");
   }
   return;
}

int dot3GetNumTex (void)
{
   return(DOT3_NUM_MAPS);
}
