/******************************************************************************
 * filename: light.cpp                                                        *
 *                                                                            *
 * author:   Christopher Oat                                                  *
 *           ATI Research, Inc.                                               *
 *           3D Application Research Group                                    *
 *           email: coat@ati.com                                              *
 *                                                                            *
 * description: uses a point sprite to show the current location of a point-  *
 *              light source.                                                 *
 *                                                                            *
 ******************************************************************************
 $Header::                                                                    $
 ******************************************************************************
 *   (C) 2001 ATI Research, Inc.  All rights reserved.                        *
 ******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <GL/gl.h>
#include <GL/glext.h>
#include <GL/glut.h>

#include "globals.h"
#include "GLTexture.h"
#include "GLErr.h"
#include "matrix.h"
#include "light.h"


static GLfloat lightPos[3] = {70.0, 0.0, 70.0};
static GLfloat lightPosES[3] = {0.0, 0.0, 0.0};
static GLfloat lightMat[16]= {1.0, 0.0, 0.0, 0.0,
                              0.0, 1.0, 0.0, 0.0,
                              0.0, 0.0, 1.0, 0.0,
                              0.0, 0.0, 0.0, 1.0};

static int lightIsInitialized=0;

static GLuint lightTextureID;
static GLuint lightListID[2];
static ATI_GLTexture *lightTexture;
static char *lightFileName = FILE_PATH "light0.tga";

static void lightFaceCamera (void);

void lightInit (void) 
{

   glGenTextures(1, &lightTextureID);

   // Reference to global error message.
   extern char gszErrMsg[];
   gszErrMsg[0] = '\0';
   
   // base map
   printf("Loading Light Texture\n");
   printf("---------------------\n");

   printf("%s",lightFileName);
   gszErrMsg[0] = '\0';

   lightTexture = new ATI_GLTexture(lightFileName, lightTextureID, GL_TRUE, GL_RGBA, GL_RGBA);
   
   if (gszErrMsg[0] != '\0')
      printf("\tError loading %s: %s\n", lightFileName, gszErrMsg);
   else
      printf("\tDone!\n");
  
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER_ARB);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER_ARB);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
   
   printf("\n");

   lightListID[0] = glGenLists(1);
   glNewList(lightListID[0], GL_COMPILE);

      glActiveTexture(GL_TEXTURE0_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE1_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE2_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

      // UNIT 0
      glActiveTexture(GL_TEXTURE0_ARB);
      glBindTexture (GL_TEXTURE_2D, lightTextureID);
      glEnable(GL_TEXTURE_2D);
      glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_ARB);
      glTexEnvf(GL_TEXTURE_ENV, GL_COMBINE_RGB_ARB, GL_REPLACE);
      glTexEnvf(GL_TEXTURE_ENV, GL_COMBINE_ALPHA_ARB, GL_REPLACE);

      glTexEnvf(GL_TEXTURE_ENV, GL_SOURCE0_RGB_ARB, GL_TEXTURE);
      glTexEnvf(GL_TEXTURE_ENV, GL_OPERAND0_RGB_ARB, GL_SRC_COLOR);

      glTexEnvf(GL_TEXTURE_ENV, GL_SOURCE0_ALPHA_ARB, GL_TEXTURE);
      glTexEnvf(GL_TEXTURE_ENV, GL_OPERAND0_ALPHA_ARB, GL_SRC_COLOR);
            
      glBegin(GL_TRIANGLE_STRIP);

      glMultiTexCoord2f(GL_TEXTURE0_ARB, 0.0, 0.0);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(-20.0, -20.0, 0.0);
      

      glMultiTexCoord2f(GL_TEXTURE0_ARB, 1.0, 0.0);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(20.0, -20.0, 0.0);

      glMultiTexCoord2f(GL_TEXTURE0_ARB, 0.0, 1.0);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(-20.0, 20.0, 0.0);

      glMultiTexCoord2f(GL_TEXTURE0_ARB, 1.0, 1.0);
      glNormal3f(0.0, 0.0, 1.0);
      glVertex3f(20.0, 20.0, 0.0);
     
      glEnd();

      glActiveTexture(GL_TEXTURE0_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE1_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE2_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);


   glEndList();


   lightListID[1] = glGenLists(1);
   glNewList(lightListID[1], GL_COMPILE);

      glActiveTexture(GL_TEXTURE0_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE1_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glActiveTexture(GL_TEXTURE2_ARB);
         glMatrixMode(GL_TEXTURE);
         glLoadIdentity ();
         glMatrixMode(GL_MODELVIEW);
         glDisable(GL_TEXTURE_3D);
         glDisable(GL_TEXTURE_CUBE_MAP_ARB);
         glDisable(GL_TEXTURE_2D);
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
         glDisable(GL_TEXTURE_GEN_R);
         glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

      glBegin(GL_TRIANGLE_STRIP);

      glNormal3f(0.0, 0.0, 1.0);
      glColor4f(1.0, 1.0, 1.0, 1.0);
      glVertex3f(-20.0, -20.0, 0.0);

      glNormal3f(0.0, 0.0, 1.0);
      glColor4f(1.0, 1.0, 1.0, 1.0);
      glVertex3f(20.0, -20.0, 0.0);

      glNormal3f(0.0, 0.0, 1.0);
      glColor4f(1.0, 1.0, 1.0, 1.0);
      glVertex3f(-20.0, 20.0, 0.0);

      glNormal3f(0.0, 0.0, 1.0);
      glColor4f(1.0, 1.0, 1.0, 1.0);
      glVertex3f(20.0, 20.0, 0.0);
      
      glEnd();

   glEndList();


   lightIsInitialized = 1;

   return;
}



// computes a transform matrix to make point sprite face the camera
static void lightFaceCamera (void)
{
   static GLfloat cam_es[3]= {0.0, 0.0, 0.0};
   static GLfloat camup_es[3]= {0.0, 1.0, 0.0};
   static GLfloat modelViewMatrix[16];
   static GLfloat modelViewMatrixInverse[16];
   

   // calculate camera location and up-vector in object space
   glGetFloatv(GL_MODELVIEW_MATRIX, modelViewMatrix);
   matrixInvert(modelViewMatrix, modelViewMatrixInverse);
   
   GLfloat cam_os[3]= {0.0, 0.0, 0.0};
   GLfloat camup_os[3]= {0.0, 0.0, 0.0};
         
   vecMatMult(cam_es, modelViewMatrixInverse, cam_os);
   vecMat3x3Mult(camup_es, modelViewMatrixInverse, camup_os);

   vecNormalize(camup_os);

   // create a transform matrix to rotate point sprite such that it
   // faces the camera with an up-vector equal to that of the camera's
   // in object space.
   GLfloat zvec[3];
   GLfloat xvec[3];

   zvec[0] = cam_os[0] - lightPos[0];
   zvec[1] = cam_os[1] - lightPos[1];
   zvec[2] = cam_os[2] - lightPos[2];
   vecNormalize(zvec);

   vecCrossProd(camup_os, zvec, xvec);
   vecNormalize(xvec);

   lightMat[0 ] = xvec[0];
   lightMat[1 ] = xvec[1];
   lightMat[2 ] = xvec[2];

   lightMat[4 ] = camup_os[0];
   lightMat[5 ] = camup_os[1];
   lightMat[6 ] = camup_os[2];

   lightMat[8 ] = zvec[0];
   lightMat[9 ] = zvec[1];
   lightMat[10] = zvec[2];


   return;

}


// display point sprite
void lightDisplay (GLboolean wireframe)
{

   if (lightIsInitialized)
   {
      static GLfloat modelViewMatrix[16];
      // calculate light's position in eye space
      glGetFloatv(GL_MODELVIEW_MATRIX, modelViewMatrix);
      vecMatMult(lightPos, modelViewMatrix, lightPosES);
      
      // compute lightMap to rotate point sprite to face camera
      lightFaceCamera();

      glEnable(GL_BLEND);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE);
 
      glPushMatrix();

      // translate point sprite to correct location then multiply by lightMat
      // to make it face the camera.
      glTranslatef(lightPos[0], lightPos[1], lightPos[2]);
      glMultMatrixf(lightMat);
      

      if (wireframe)
      {
         glCallList(lightListID[1]);
      }
      else
      {
         glCallList(lightListID[0]);
      }
      
      glPopMatrix();

      glDisable(GL_BLEND);
   }
   else
      printf("Error: call to lightDisplay() without prior lightInit()\n");

   return;

}

// give light position in eye space
void lightGetPos (GLfloat vec[3])
{
   vecCopy(lightPosES, vec);
   return;
}
