/*!
 * @file collision_filter.h
 * @brief Definition of a collision filter: checks if a certain WorldEntity is responsive for another WorldEntity
 *
*/

#ifndef _COLLISION_FILTER_H
#define _COLLISION_FILTER_H

#include "base_object.h"
#include "cr_engine.h"

#include <vector>
#include <list>


class Collision;
class WorldEntity;

namespace CoRe
{

  class CollisionReaction;


  /**
   * A class for defining collision reactions and storing events (functional object)
   *
   * This class basically checks if the owner of this filter (a WorldEntity) is responsive for a certain other WorldEntity.
   * The check is performed via the operator() (therefore it's a functional objects). For each CollisionReaction there is a list
   * of WorldEntities (their ClassIDs) to which it listens to.
   */
  class CollisionFilter : public BaseObject
  {
    ObjectListDeclaration(CollisionFilter);


    /* Constructor/Deconstructors */
  public:
    CollisionFilter(WorldEntity* owner);
    virtual ~CollisionFilter();

  private:
    CollisionFilter(const CollisionFilter& collisionFilter) {}
    WorldEntity*                  _owner;                   //!< the worldenity this reaction will be applied on


  public:
    /* Defines Operators */
    bool operator()(const WorldEntity& entity) const;
    bool operator()(const WorldEntity& entity, const CREngine::ReactionType type) const;


    /* Collision Reaction subscription unsubscription Block */
  public:
    void subscribeReaction(CREngine::ReactionType type, const ClassID& target1);
    void subscribeReaction(CREngine::ReactionType type, const ClassID& target1, const ClassID& target2);
    void subscribeReaction(CREngine::ReactionType type, const ClassID& target1, const ClassID& target2, const ClassID& target3);

    void unsubscribeReaction(CREngine::ReactionType type);
    void unsubscribeReactions();

    inline bool isSubscribed(CREngine::ReactionType type) const { /*if( this->validCRType(type)) return this->_filters[type] != NULL; else return false;*/}

  private:
    std::vector<ClassID>          _filters[CREngine::CR_NUMBER];  //!< an array of filter targets: for each collision type a list of filter objects


    /* Misc State Informations */
  public:
    /** @returns true if this handle should be pulled also if there are no collisions, can also be set with this function (reference)*/
    inline bool& bContinousPoll() { return this->_bContinuousPoll; }
    /** @returns true if this filter should be pulled always */
    inline bool bContinousPoll() const { return this->_bContinuousPoll; }
    /** @returns true if this @param type Collision Reaction type is a valid number */
    inline bool validCRType(const CREngine::ReactionType& type) const {return (type >= 0 && type < CREngine::CR_NUMBER); }
    /** @returns true if this filter is reactive i.e. at least one filter criterion installed */
    inline bool bReactive() const { return this->_bReactive; }

  private:
    bool                          _bContinuousPoll;         //!< if this is true
    bool                          _bStopOnFirstCollision;   //!< true if the cd of this object should be terminated after one match
    bool                          _bReactive;               //!< true if this class has at least one filter criterion == one target

  };

}
#endif /* _COLLISION_FILTER_H */
