/*
   orxonox - the future of 3D-vertical-scrollersf

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_COLLISION_REACTION

#include "collision_filter.h"

#include "world_entity.h"

#include "collision.h"
#include "collision_event.h"
#include "collision_reaction.h"

#include "cr_object_damage.h"
#include "cr_physics_ground_walk.h"
#include "cr_physics_full_walk.h"

#include "debug.h"

#include <vector>


namespace CoRe
{

  ObjectListDefinition(CollisionFilter);

  /**
   * standard constructor
   * @todo this constructor is not jet implemented - do it
  */
  CollisionFilter::CollisionFilter (WorldEntity* owner)
  {
    this->registerObject(this, CollisionFilter::_objectList);

    this->_owner = owner;

    this->_bContinuousPoll = false;
    this->_bStopOnFirstCollision = false;
    this->_bReactive = false;
  }


  /**
   * standard deconstructor
  */
  CollisionFilter::~CollisionFilter ()
  {
    this->unsubscribeReactions();
  }


  /**
   * subscribe reaction
   */
  void CollisionFilter::subscribeReaction(CoRe::CREngine::ReactionType type, const ClassID& target1)
  {
    // check if its a valid type
    if( likely(this->validCRType( type)))
    {
      //check if this class isn't already registered
      std::vector<ClassID>::iterator it = this->_filters[type].begin();
      for(; it != this->_filters[type].end(); it++)
      {
        if( unlikely(*it == target1))
          return;
      }

      // so subscribe the reaction finally
      this->_filters[type].push_back(target1);
      this->_bReactive = true;
    }
  }


  /**
   * subscribe for a reaction
   */
  void CollisionFilter::subscribeReaction(CoRe::CREngine::ReactionType type, const ClassID& target1, const ClassID& target2)
  {
    this->subscribeReaction(type, target1);
    this->subscribeReaction(type, target2);
  }


  /**
   * subscribe for a reaction
   */
  void CollisionFilter::subscribeReaction(CoRe::CREngine::ReactionType type, const ClassID& target1, const ClassID& target2, const ClassID& target3)
  {
    this->subscribeReaction(type, target1);
    this->subscribeReaction(type, target2);
    this->subscribeReaction(type, target3);
  }

  /**
   * unsubscribe from a specific collision reaction
   */
  void CollisionFilter::unsubscribeReaction(CoRe::CREngine::ReactionType type)
  {
    if( likely(this->validCRType( type)))
      this->_filters[type].clear();
  }

  /**
   * unsubscribe from all collision reactions
   */
  void CollisionFilter::unsubscribeReactions()
  {
    for(int i = 0; i < CREngine::CR_NUMBER; i++)
      this->_filters[i].clear();
  }



  /**
   * tests if the owner WorldEntity is listening to collisions from another specif WorldEntity entity
   *  @param entity WorldEntity to test against
   *
   * This is the most important interface function of this class: it performs a check and returns true
   * if the WorldEntity entity is actualy responsive for a certain other WorldEntity
   */
  bool CollisionFilter::operator()(const WorldEntity& entity) const
  {
    // if there are no installed criterions just ommit and
    if( this->bReactive())
      return false;

    // goes through all registered filter criterions and looks for matches
    for( int i = 0; i < CREngine::CR_NUMBER; i++ )
    {
      std::vector<ClassID>::const_iterator it = this->_filters[i].begin();
      for(; it != this->_filters[i].end(); i++ )
        if( unlikely(entity.isA(*it)))
          return true;
    }

    return false;
  }


  /**
  * tests if the owner WorldEntity is listening to collisions from another specif WorldEntity entity
  *  @param entity WorldEntity to test against
  *
  * This is the most important interface function of this class: it performs a check and returns true
  * if the WorldEntity entity is actualy responsive for a certain other WorldEntity
   */
  bool CollisionFilter::operator()(const WorldEntity& entity, const CREngine::ReactionType type) const
  {
    // if there are no installed criterions just ommit and
    if( this->bReactive())
      return false;

    // goes through all registered filter criterions and looks for matches
    std::vector<ClassID>::const_iterator it = this->_filters[type].begin();
    for(; it != this->_filters[type].end(); it++ )
      if( unlikely(entity.isA(*it)))
        return true;

    return false;
  }



} // namespace end



