/*!
 * @file collision.h
 *  Definition of a collision relation of two WorldEntities
 *
 *  Is shared between two WorldEntity's CollisionHandles if both are subscribed to this event. In this case only one
 *  of the two CollisionHandles will calculate the CollisionReaction and the bDispatched flag will be set afterwards
 *  to signal that it's already cared about and should be ignored.
 *
 *  The collisions itself are saved in this container (CollisionEvent). Since there can be multiple collision events
 *  for one collision. Imagine: two objects are intersecting (this throws a Collision): many collision boxes will fire
 *  each of this boxes will "create" a CollisionEvent.
 */

#ifndef _COLLISION_H
#define _COLLISION_H

#include "vector.h"
#include <vector>



class WorldEntity;
class BoundingVolume;

namespace CoRe
{

  class CollisionEvent;

  //! A class representing a simple collision
  class Collision
  {
  public:

    Collision();
    virtual ~Collision();

    /** collides two WorldEntities @param entityA world entity A, @param entityB world entity B, @param bvA volume A @param bvB volumeB */
    inline void collide(WorldEntity* entityA, WorldEntity* entityB) { this->entityA = entityA; this->entityB = entityB; this->bDispatched = false; }


    /* list stuff */
  public:
    typedef std::vector<CollisionEvent*>                  collisionVector;  //!< the vector containing all collision events
    typedef std::vector<CollisionEvent*>::iterator        iterator;         //!< iterator definition
    typedef std::vector<CollisionEvent*>::const_iterator  const_iterator;   //!< constant iterator definition

    /** registers a @param event CollisionEvent to take place */
    inline void registerCollisionEvent(CollisionEvent* event) { this->_collisionEvents.push_back(event); this->bDispatched = false;}
    /** @returns a vector of collision events */
    inline const std::vector<CollisionEvent*>& getCollisionEvents() const { return this->_collisionEvents; }

    void flushCollisionEvents();


    /** @returns iterator pointing to the beginning of the collision event vector */
    inline iterator       begin()        { return this->_collisionEvents.begin(); }
    /** @returns const iterator pointing to the beginning of the collision event vector */
    inline const_iterator begin() const  { return this->_collisionEvents.begin(); }
    /** @returns iterator pointing to the end of the collision event vector */
    inline iterator       end()          { return this->_collisionEvents.end(); }
    /** @returns const iterator pointing to the end of the collision event vector */
    inline const_iterator end() const    { return this->_collisionEvents.end(); }


  private:
    collisionVector  _collisionEvents;               //!< the collision event list



    /* misc interface functions */
  public:
    /** @return Collision WorldEntity A */
    inline WorldEntity* getEntityA() const { return this->entityA; }
    /** @return Collision WorldEntity B */
    inline WorldEntity* getEntityB() const { return this->entityB; }
    inline bool match(const WorldEntity& entityA, WorldEntity& entityB) const {
      return ((this->entityA == &entityA && this->entityB == &entityB) || (this->entityA == &entityB && this->entityB == &entityA)); }

    /** @return true if Entity A collides */
    inline bool isEntityACollide() const { return this->entityACollide; }
    /** sets the flag if it reacts @param flag true if it should react on entityA*/
    inline void setEntityACollide(bool flag) { this->entityACollide = flag; }
    /** @return true if Entity B collides */
    inline bool isEntityBCollide() const { return this->entityBCollide; }
    /** sets the flag if it reacts @param flag true if it should react on entityB*/
    inline void setEntityBCollide(bool flag) { this->entityACollide = flag; }



    /** @returns true if this Collision has already been dispatched */
    inline bool isDispatched() { return this->bDispatched; }
    /** sets the dispatched flag to true */
    inline void dispatched() { this->bDispatched = true; }

  private:
    WorldEntity*                 entityA;                       //!< the collision body A
    WorldEntity*                 entityB;                       //!< the collision body B
    bool                         entityACollide;                //!< true if entity A is subscribed for collision reaction
    bool                         entityBCollide;                //!< true if entity B is subscribed for collision reaction

    bool                         bDispatched;                   //!< true if this collision has already been dispatched

  };
}

#endif /* _COLLISION_H */
