/*
* Original Tutorial by Jeff Molofee
* (ported to Linux/SDL by Ti Leggett '01)
*
* Visit Jeff at http://nehe.gamedev.net/
*/

#include <GL/gl.h>
#include <GL/glu.h>
#include "SDL.h"

/* screen width, height, and bit depth */
#define SCREEN_WIDTH  640
#define SCREEN_HEIGHT 480
#define SCREEN_BPP     32

/* This is our SDL surface */
SDL_Surface *surface;

/* Prototype declaration */
int resizeWindow(int width, int height);
int drawGLScene();


int main( int argc, char **argv )
{
  /*================================ SDL INIT ============================================ */
  // Wir automatisch in der GraphicsEngine gemacht
  // siehe src/lib/graphics/graphics_engine.cc

  /* Flags to pass to SDL_SetVideoMode */
  int videoFlags;
  /* main loop variable */
  int done = false;
  /* used to collect events */
  SDL_Event event;
  /* this holds some info about our display */
  const SDL_VideoInfo *videoInfo;


  /* initialize SDL */
  if (SDL_Init( SDL_INIT_VIDEO ) < 0)
  {
    fprintf(stderr, "Video initialization failed: %s\n", SDL_GetError());
    exit(1);
  }

  /* Fetch the video info */
  videoInfo = SDL_GetVideoInfo();

  if(!videoInfo)
  {
    fprintf(stderr, "Video query failed: %s\n", SDL_GetError());
    exit(1);
  }

  /* the flags to pass to SDL_SetVideoMode */
  videoFlags  = SDL_OPENGL;          /* Enable OpenGL in SDL */
  videoFlags |= SDL_GL_DOUBLEBUFFER; /* Enable double buffering */
  videoFlags |= SDL_HWPALETTE;       /* Store the palette in hardware */
  videoFlags |= SDL_RESIZABLE;       /* Enable window resizing */

  /* This checks to see if surfaces can be stored in memory */
  if(videoInfo->hw_available)
    videoFlags |= SDL_HWSURFACE;
  else
    videoFlags |= SDL_SWSURFACE;

  /* This checks if hardware blits can be done */
  if (videoInfo->blit_hw)
    videoFlags |= SDL_HWACCEL;

  /* Sets up OpenGL double buffering */
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

  /* get a SDL surface */
  surface = SDL_SetVideoMode( SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_BPP,
                              videoFlags );

  /* Verify there is a surface */
  if(!surface)
  {
    fprintf(stderr,  "Video mode set failed: %s\n", SDL_GetError());
    exit(1);
  }

  /* initialize OpenGL */
  /* Set the background black */
  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

  /* Enables Depth Testing */
  glEnable( GL_DEPTH_TEST );


  /* resize the initial window */
  resizeWindow(SCREEN_WIDTH, SCREEN_HEIGHT);


  /*================================ MAIN-LOOP ============================================ */
  // Entspricht der Orxonox-Main-Loop in
  // src/story_entities/game_world.cc:291
  // ( void GameWorld::run() )


  /* wait for events */
  while(!done)
  {
    /* handle the events in the queue */

    while(SDL_PollEvent(&event))
    {
      switch( event.type )
      {
      case SDL_VIDEORESIZE:
        /* handle resize event */
        surface = SDL_SetVideoMode(event.resize.w, event.resize.h, 16, videoFlags);
        if(!surface)
        {
          fprintf(stderr, "Could not get a surface after resize: %s\n", SDL_GetError());
          exit(1);
        }
        resizeWindow(event.resize.w, event.resize.h);
        break;
      case SDL_QUIT:
        /* handle quit requests */
        done = true;
        break;
      default:
        break;
      }
    }

    // hier wird das ganze gezeichnet!
    drawGLScene();
  }

  /* clean up the window */
  SDL_Quit( );

  /* Should never get here */
  return( 0 );
}


/* Here goes our drawing code */
int drawGLScene()
{
  /* rotational vars for the triangle and quad, respectively */
  static GLfloat rtri;
  /* These are to calculate our fps */
  static GLint T0     = 0;
  static GLint Frames = 0;


  /* Clear The Screen And The Depth Buffer */
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);


  /* ================================== WorldEntity zeichnen ======================================== */
  // das ist der code, der in eine WorldEntity::draw() funktion passen wuerde


  /* ================================== WorldEntity zeichnen fertig ================================ */


  /* Draw it to the screen */
   SDL_GL_SwapBuffers();

  /* Gather our frames per second */
  Frames++;
  {
    GLint t = SDL_GetTicks();
    if (t - T0 >= 5000) {
      GLfloat seconds = (t - T0) / 1000.0;
      GLfloat fps = Frames / seconds;
      printf("%d frames in %g seconds = %g FPS\n", Frames, seconds, fps);
      T0 = t;
      Frames = 0;
    }
  }

  /* Increase The Rotation Variable For The Triangle ( NEW ) */
  rtri  += 0.2f;

  return(true);
}



/* function to reset our viewport after a window resize */
int resizeWindow(int width, int height)
{
  /* Height / width ration */
  GLfloat ratio;

  /* Protect against a divide by zero */
  if(height == 0)
    height = 1;

  ratio = (GLfloat)width / (GLfloat)height;

  /* Setup our viewport. */
  glViewport(0, 0, (GLsizei)width, (GLsizei)height);

  /* change to the projection matrix and set our viewing volume. */
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  /* Set our perspective */
  gluPerspective(90.0f, ratio, 0.1f, 100.0f);

  /* Make sure we're chaning the model view and not the projection */
  glMatrixMode(GL_MODELVIEW);

  /* Reset The View */
  glLoadIdentity();

  return(true);
}


