/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Christoph Renner
   co-programmer: ...

*/

#include "spectator.h"

#include "src/lib/util/loading/factory.h"
#include "key_mapper.h"

#include "shared_network_data.h"

#include "src/world_entities/creatures/fps_player.h"

#include "class_id_DEPRECATED.h"
ObjectListDefinitionID(Spectator, CL_SPECTATOR);
CREATE_FACTORY(Spectator);


/**
 *  destructs the Spectator, deletes alocated memory
 */
Spectator::~Spectator ()
{
  this->setPlayer(NULL);
}


/**
 *  creates a new Spectator from Xml Data
 * @param root the xml element containing Spectator data

   @todo add more parameters to load
 */
Spectator::Spectator(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);

}


/**
 * initializes a Spectator
 */
void Spectator::init()
{
//  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->registerObject(this, Spectator::_objectList);

  this->getWeaponManager().changeWeaponConfig(1);

  this->bLeft = false;
  this->bRight = false;
  this->bForward = false;
  this->bBackward = false;
  this->xMouse = 0.0f;
  this->yMouse = 0.0f;

  this->setHealthMax(100);
  this->setHealth(80);

  this->mouseDir = this->getAbsDir();

  //add events to the eventlist
  registerEvent(KeyMapper::PEV_FORWARD);
  registerEvent(KeyMapper::PEV_BACKWARD);
  registerEvent(KeyMapper::PEV_LEFT);
  registerEvent(KeyMapper::PEV_RIGHT);
  registerEvent(KeyMapper::PEV_FIRE1);
  registerEvent(KeyMapper::PEV_JUMP);
  registerEvent(EV_MOUSE_MOTION);

  this->getWeaponManager().setSlotCount(0);

  this->getWeaponManager().getFixedTarget()->setParent(this);
  this->getWeaponManager().getFixedTarget()->setRelCoor(100000,0,0);

  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);


  registerVar( new SynchronizeableBool( &bLeft, &bLeft, "bLeft", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bRight, &bRight, "bRight", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bForward, &bForward, "bForward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bBackward, &bBackward, "bBackward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableQuaternion( &mouseDir, &mouseDir, "mouseDir", PERMISSION_OWNER ) );
}


/**
 * loads the Settings of a Spectator from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void Spectator::loadParams(const TiXmlElement* root)
{
  Playable::loadParams(root);
}

void Spectator::setPlayDirection(const Quaternion& quat, float speed)
{
  this->mouseDir = quat;
  this->angleY = quat.getHeading();
  this->angleX = quat.getAttitude();
}


void Spectator::reset()
{
  this->bLeft = false;
  this->bRight = false;
  this->bForward = false;
  this->bBackward = false;
  this->xMouse = 0.0f;
  this->yMouse = 0.0f;

  this->setHealth(80);
}


void Spectator::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false );
  this->attachCamera();
}

void Spectator::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);
  this->detachCamera();
}


/**
 *  this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 * @todo dont let Spectator fly through walls
 */
void Spectator::collidesWith(WorldEntity* entity, const Vector& location)
{
}



/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
 */
void Spectator::tick (float time)
{
  Playable::tick( time );

  if( ( xMouse != 0 || yMouse != 0 ) && this->getOwner() == SharedNetworkData::getInstance()->getHostID() )
  {
    xMouse *= time / 10;
    yMouse *= time / 10;

    angleX -= xMouse;
    angleY -= yMouse;

    if ( angleY > 2.05 )
      angleY = 2.05;

    if ( angleY < -1.15 )
      angleY = -1.15;

    this->mouseDir = Quaternion( angleX, Vector( 0, 1, 0 ) ) * Quaternion( angleY, Vector( 0, 0, 1 ) );

    xMouse = yMouse = 0;
  }

  this->setAbsDir( this->mouseDir );

  Vector velocity;

  if ( this->bForward )
  {
    velocity += this->getAbsDirX();
  }

  if ( this->bBackward )
  {
    velocity -= this->getAbsDirX();
  }

  if ( this->bRight )
  {
    velocity += this->getAbsDirZ();
  }

  if ( this->bLeft )
  {
    velocity -= this->getAbsDirZ();
  }

  velocity *= 100;

  this->shiftCoor( velocity*time );
}

/**
 * @todo switch statement ??
 */
void Spectator::process(const Event &event)
{
  Playable::process(event);

  if( event.type == KeyMapper::PEV_LEFT)
    this->bLeft = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
    this->bRight = event.bPressed;
  else if( event.type == KeyMapper::PEV_FORWARD)
    this->bForward = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_BACKWARD)
    this->bBackward = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {
    this->xMouse += event.xRel;
    this->yMouse += event.yRel;
  }
  else if( event.type == KeyMapper::PEV_JUMP)
  {
    FPSPlayer * fps = new FPSPlayer();
    //WorldEntity * fps = new WorldEntity();
    
    fps->setAbsCoor( this->getAbsCoorX(), this->getAbsCoorY(), this->getAbsCoorZ() );
    fps->setAbsDir( this->getAbsDir() );
    fps->loadMD2Texture( "doom_guy.png" );
    fps->loadModel( "models/creatures/doom_guy.md2", 10.0f );
    //fps->loadModel( "models/ships/terran_cruizer.obj" );

    //((Playable*)fps)->setPlayDirection(  0, 0, 1, 0 );
  }
}




