/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Benjamin Grauer
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_BASE

#include "base_object.h"

#include "util/debug.h"
#include "util/loading/load_param.h"

ObjectListDefinition(BaseObject);

/**
 * @brief sets the name from a LoadXML-Element
 * @param objectName: The name of the Object.
 */
BaseObject::BaseObject(const std::string& objectName)
{
  this->className = "BaseObject";

  this->objectName = objectName;
  this->xmlElem = NULL;
  this->registerObject(this, BaseObject::_objectList);
}

/** 
 * copyconstructor
 * @param bo instance to copy
 */
BaseObject::BaseObject( const BaseObject& bo )
{
  this->className = "BaseObject";
  this->objectName = bo.objectName;
  this->xmlElem = (bo.xmlElem)? bo.xmlElem->Clone() : NULL;
  this->registerObject( this, BaseObject::_objectList);
}


/**
 * @brief standard deconstructor
*/
BaseObject::~BaseObject ()
{
  /// Remove from the ObjectLists
  ClassEntries::iterator it;
  PRINTF(5)("Deleting Object of type %s::%s\n", this->getClassCName(), getCName());
  for (it = this->_classes.begin(); it != this->_classes.end(); ++it)
  {
    //if (ORX_DEBUG >= 5)
      assert((*it)._objectList->checkIteratorInList((*it)._iterator) || (*it)._objectList->checkObjectInList(this));
    (*it)._objectList->unregisterObject((*it)._iterator);
    delete (*it)._iterator;
  }

  if (this->xmlElem != NULL)
    delete this->xmlElem;
}

/**
 * @brief loads parameters
 * @param root the element to load from
 */
void BaseObject::loadParams(const TiXmlElement* root)
{
  // all loadParams should arrive here, and be tested for (root != NULL)
  assert (root != NULL);

  // copy the xml-element for to know how it was loaded.
  if (this->xmlElem != NULL)
    delete this->xmlElem;
  this->xmlElem = root->Clone();

  // name setup
  LoadParam(root, "name", this, BaseObject, setName)
      .describe("the Name of the Object.");
}

/**
 * @brief set the name of the Object
 * @param objectName The new name of the Object.
 */
void BaseObject::setName (const std::string& objectName)
{
  this->objectName = objectName;
}


/**
 * @brief Seeks in the Inheritance if it matches objectList.
 * @param objectList The ObjectList this should be a member of (by Pointer-comparison).
 * @return True if found, false if not.
 */
bool BaseObject::isA(const ObjectListBase& objectList) const
{
  ClassEntries::const_iterator it;
  for (it = this->_classes.begin(); it != this->_classes.end(); ++it)
    if ((*it)._objectList == &objectList)
      return true;
  return false;
}


/**
 * @brief Seeks in the Inheritance if it matches objectList.
 * @param classID The ClassID this should be a member of (by Pointer-comparison).
 * @return True if found, false if not.
 */
bool BaseObject::isA(const ClassID& classID) const
{
  ClassEntries::const_iterator it;
  for (it = this->_classes.begin(); it != this->_classes.end(); ++it)
    if (*(*it)._objectList == classID)
      return true;
  return false;
}

/**
 * @brief Seeks in the Inheritance if it matches objectList.
 * @param classID The ClassID of the class this should be a member of.
 * @return True if found, false if not.
 */
bool BaseObject::isA(int classID) const
{
  ClassEntries::const_iterator it;
  for (it = this->_classes.begin(); it != this->_classes.end(); ++it)
    if (*(*it)._objectList == classID)

      return true;
  return false;
}

/**
 * @brief Seeks in the Inheritance if it matches objectList.
 * @param className The ClassName of the class this should be a member of.
 * @return True if found, false if not.
 */
bool BaseObject::isA(const std::string& className) const
{
  ClassEntries::const_iterator it;
  for (it = this->_classes.begin(); it != this->_classes.end(); ++it)
    if (*(*it)._objectList == className)
      return true;
  return false;
}


/**
 * @brief This is for debug purposes, to see the Inheritances of this Object and its classes.
 *
 * The Inheritance will be listed in a Linear fashion, diamand structures are resolved in a linear dependency.
 */
void BaseObject::listInheritance() const
{
  PRINT(0)("Listing inheritance diagram for %s::%s: ", getClassCName(), getCName());
  ClassEntries::const_iterator it;
  for (it = this->_classes.begin(); it != this->_classes.end(); ++it)
    PRINT(0)(" -> %s(id:%d)", (*it)._objectList->name().c_str(), (*it)._objectList->id());
  PRINT(0)("\n");

}
