/*!
 * @file object_list.h
 * @brief Definition of a dynamically allocating ClassID
 *
 */

#ifndef _OBJECT_LIST_H
#define _OBJECT_LIST_H

#include "class_id.h"
#include <map>
#include <list>
#include <string>

#include <cassert>
#include <iostream>

/**
 * @brief Use this macro to easily declare a Class to store its own ObjectListDeclaration
 * @param ClassName the Name of the Class.
 * @note: Using this inside of a Class means, that you loose the member: _objectList, while defining
 * two new functions objectList() and classID().
 */
#define ObjectListDeclaration(ClassName) \
  public: \
   static inline const ObjectList<ClassName>& objectList() { return ClassName::_objectList; }; \
   static inline const ClassID& staticClassID() { return ClassName::_objectList.identity(); }; \
  private: \
   static ObjectList<ClassName> _objectList

/**
 * @brief Use this macro to easily define a Class to store its own ObjectListDefinition
 * @param ClassName: the Name of the Class.
 * @param ID: optional set a Fixed ID.
 */
#define ObjectListDefinitionID(ClassName, ID) \
   ObjectList<ClassName> ClassName::_objectList(#ClassName, ID)

/**
 * @brief Use this macro to easily define a Class to store its own ObjectListDefinition
 * @param ClassName: the Name of the Class.
 */
#define ObjectListDefinition(ClassName) \
    ObjectListDefinitionID(ClassName, -1)

class BaseObject;
//! The superclass that all ObjectLists follow.
/**
 * @see template<class T> ObjectList<T>
 */
class ObjectListBase
{
public:
  /** @brief A Typedefinition for the Base-List that can be retrieved with getBaseObjectList(base_list*) */
  typedef std::list<BaseObject*>        base_list;
  /** @brief An iterator for the base_list */
  typedef base_list::iterator           base_iterator;
  //! A fast iterator Base-Class, for iterator-casting and storing.
  /** @note This Iterator is explicitely used only for storage purposes in the BaseObject */
  class IteratorBase { };

public:
  /** @returns The Identity of the Class stored within. */
  inline const ClassID&                 identity() const { return _identity; }
  /** @returns the ID of the Identity of the ObjectList */
  inline int                            id() const { return _id; };
  /** @returns The Name of the Class stored in this ObjectList */
  inline const std::string&             name() const { return _name; };
  /** @param id The id to compare @returns true on match, false otherwise */
  inline bool                           operator==(int id) const { return _id == id; };
  /** @param id The id to compare @returns true on match, false otherwise */
  inline bool                           operator==(const ClassID& id) const { return id == _id; };
  /** @param name The name to compare @returns true on match, false otherwise */
  inline bool                           operator==(const std::string& name) const { return _name == name; };
  /** @param id The id to acquire @param name the Name to acquire @brief stores a Name and an ID in a pointer. */
  inline void                           acquireID(const int*& id, const std::string*& name) const { id = &_id; name = &_name; };
  /** @brief fills a list of Objects into a BaseObject*-List. @param list the list to fill */
  virtual void                          getBaseObjectList(base_list* list) const = 0;

  static const ClassID&                 retrieveIdentity(int id);
  static const ClassID&                 retrieveIdentity(const std::string& name);

  static const ObjectListBase* const    getObjectList(int classID);
  static const ObjectListBase* const    getObjectList(const std::string& className);
  static const ObjectListBase* const    getObjectList(const ClassID& classID);

  static BaseObject*                    getBaseObject(int classID, const std::string& objectName);
  static BaseObject*                    getBaseObject(const std::string& className, const std::string& objectName);
  static BaseObject*                    getBaseObject(const ClassID& classID, const std::string& objectName);

  /** @returns an Object with Name name out of this List @param name the name of the Object. */
  virtual BaseObject*                   getBaseObject(const std::string& name) const = 0;

  static const std::list<std::string>&  getClassNames();


  static unsigned int                   classCount();
  void                                  debug(unsigned int level) const;
  static void                           debugAll(unsigned int level);

  static const std::string&             IDToString(int classID);
  static int                            StringToID(const std::string& className);

  //! Only used to unsubscribe a BaseObject. (just try to make a valid iterator, stupid :))
  virtual void                          unregisterObject(IteratorBase* _iterators) = 0;
  //! Only for debug purposes
  virtual bool                          checkIteratorInList(IteratorBase* _iterator) const = 0;
  virtual bool                          checkObjectInList(BaseObject* obj) const = 0;

protected:
  ObjectListBase(const std::string& className, int id = -1);
  virtual ~ObjectListBase();

private:
  /** @brief hidden copy constructor. */
  ObjectListBase(const ObjectListBase&) {};

  static bool                         classIDExists(int id);
  static bool                         classNameExists(const std::string& className);


protected:
  typedef std::map<int, ObjectListBase*>         IDMap;   //!< The Generic Map.
  typedef std::map<std::string, ObjectListBase*> NameMap; //!< The Generic Map.

private:
  int                           _id;                //!< The Unique ID of the Class.
  const std::string             _name;              //!< The Name of the Class.
  ClassID                       _identity;          //!< The Identity of the Class. (equal to _id and _name)

private:
  static IDMap*                 _classesByID;       //!< A Map of all the classes in existance.
  static NameMap*               _classesByName;     //!< A Map of all the classes in existance.
  static std::list<std::string> _classNames;        //!< A list of all the registered ClassNames.
  
public:
  static void replaceIDMap( const std::map<std::string, int>& str2id );
  static std::map<std::string, int>* createStrToId();
};


/////////////////////////
//// TEMPLATISATION /////
/////////////////////////
//! Defines a ObjectList handler for objects of type T.
/**
 * The ObjectList is a generic way to store every object created from a Class 'T'
 *  in a List. The list can be retrieved, and used constantly over iterators,
 *  as with normal std::list.
 *
 * Furthermore the linkage over the single Lists is given over the Superclass ObjectListBase.
 *
 * The approach of ObjectList is an intrusive one, meaning, that each Class that wants to
 * support it must implement a Declaration and a Definition for the ObjectList, as mentioned
 * in the next statement:
 *
 * To define a Class with a ObjectList, you have to:
 *  1. Include 'ObjectListDeclaration(T);' in its Declaration (at the beginning)
 *  2. Include 'ObjectListDefinition(T);' in some Definition file (cc-file)
 *  3. In the constructor add 'registerObject(this, objectList);'
 *
 * @note The Class must define the compare with const std::string& operator for this to work.
 *  The operator==(const std::string& name) should compare the object's name with name.
 *
 *
 *
 * Limitations:
 *  ObjectList cannot be used with other Factory style Classes, if the class is also a BaseObject,
 *   and not loaded before the ObjectList.
 *
 * example: Iterating: Iteration is made easy, and fast as follows:
 *   for (ObjectList<PlayerStats>::const_iterator it = PlayerStats::objectList().begin();
 *      it != PlayerStats::objectList().end();
 *     ++it)
 *   {
 *    (*it)->dosomething
 *   }
 *
 * example: Find an Object:
 * Playable* playable = Playable::objectList("orxonox-super-rocket-fighter"); // searches an Object By its name.
 *
 */
template<class T>
class ObjectList : public ObjectListBase
{
public:
  typedef std::list<T*>                  list;             //!< The list of Type T* (used for Objects in this ObjectList)
  typedef typename list::iterator        iterator;         //!< The iterator for the List of type T (use with ObjectList<Type>::iterator)
  typedef typename list::const_iterator  const_iterator;   //!< A constant iterator for the List of type T (use with ObjectList<Type>::const_iterator)


  //! An iterator to store Objects in the BaseObject, and remove Objects fast.
class Iterator : public ObjectListBase::IteratorBase
  {
  public:
    /** @brief creates an Iterator fast. @param it the Iterator. */
    inline Iterator(iterator it) { _it = it; }
    /** @returns the Iterator */
    inline iterator& it() { return _it; }
  private:
    typename ObjectList::iterator _it;  //!< Stored Iterator
  };

public:
  ObjectList(const std::string& name, int id = -1);
  ~ObjectList();

  virtual BaseObject*                getBaseObject(const std::string& name) const;
  T*                                 getObject(const std::string& name) const;
  /** @returns A constant list of Objects of this Class. */
  inline const list&                 objects() const { return _objects; };
  bool                               exists(const T* const object) const;

  /** @returns an Iterator to the beginning of the List. */
  inline iterator                    begin() { return _objects.begin(); };
  /** @returns a constant Iterator to the beginning of the List. */
  inline const_iterator              begin() const { return _objects.begin(); };
  /** @returns an Iterator to the end of the List. */
  inline iterator                    end() { return _objects.end(); };
  /** @returns a constant Iterator to the beginning of the List. */
  inline const_iterator              end() const { return _objects.end(); };

  /** @returns true if the List is empty. */
  inline bool                        empty() const { return _objects.empty(); };
  /** @returns the size of the List */
  inline int                         size() const { return _objects.size(); };
  /** @returns the frontmost Element of the list (normaly the last added Object). */
  inline T*                          front() const { return _objects.front(); };
  /** @returns the last added Object */
  inline T*                          back() const { return _objects.back(); };


  ObjectListBase::IteratorBase*      registerObject(T* object);
  virtual void                       unregisterObject(IteratorBase* iterator);
  bool                               checkIteratorInList(IteratorBase* iterator) const;
  bool                               checkObjectInList(BaseObject* obj) const;
protected:
  virtual void                       getBaseObjectList(ObjectListBase::base_list* list) const;


private:
  //! the copy constructor will be hidden.
  ObjectList(const ObjectList& definer) {};

private:
  list                _objects;     //!< The List of stored Objects of Type T.
};





/////////////////////////
//// IMPLEMENTATION /////
/////////////////////////
/**
 * @brief creates a new ObjectList
 * @param name The name of the Class.
 * @param id The ID of the class if desired, or -1 if an id should be assigned automatically.
 */
template <class T>
ObjectList<T>::ObjectList(const std::string& name, int id)
    : ObjectListBase(name, id)
{}

/**
 * @brief deletes the ObjectList.
 */
template <class T>
ObjectList<T>::~ObjectList()
{
  if (!_objects.empty())
  {
    // std::cout << "There are " << this->size() << " objects from class " << this->name() << "(id:" << this->id() << ") in existance\n";
  }
}

/**
 * @brief Retrieves a BaseObject matching the Name name in this List.
 * @param name the Name of the Object.
 * @returns a BaseObject pointing to the object if found, NULL otherwise.
 */
template <class T>
BaseObject* ObjectList<T>::getBaseObject(const std::string& name) const
{
  return this->getObject(name);
}



/**
 * @brief Retrieves an Object of type T matching the Name name in this List.
 * @param name the Name of the Object.
 * @returns an Object of type T pointing to the object if found, NULL otherwise.
 */
template <class T>
T* ObjectList<T>::getObject(const std::string& name) const
{
  const_iterator it;
  for (it = this->_objects.begin(); it != this->_objects.end(); ++it)
    if ((*it)->getName() == name)
      return (*it);
  return NULL;
}

/**
 * @brief checks if Object object exists in this ClassList.
 * @param object the Object to check for
 * @returns True if the object is found within the List, false otherwise
 */
template <class T>
bool ObjectList<T>::exists(const T* const object) const
{
  return (std::find(_objects.begin(), _objects.end(), object) != _objects.end());
}


/**
 * @brief retrieves a List of BaseObjects
 * @param list the list to push the ObjectList into.
 */
template <class T>
void ObjectList<T>::getBaseObjectList(ObjectListBase::base_list* list) const
{
  assert (list != NULL);
  const_iterator it;
  for (it = this->_objects.begin(); it != this->_objects.end(); ++it)
    list->push_back(*it);
}


/**
 * @brief registers an Object to the ObjectList.
 * @param object The Object to register.
 * @returns a pointer to the iterator inside of the list.
 */
template <class T>
ObjectListBase::IteratorBase* ObjectList<T>::registerObject(T* object)
{
  this->_objects.push_back(object);
  return new Iterator(--this->_objects.end());
}

/**
 * @brief removes an Object from the ClassList.
 * @param iterator the Position at which to remove the Object.
 */
template <class T>
void ObjectList<T>::unregisterObject(IteratorBase* iterator)
{
  this->_objects.erase(static_cast<Iterator*>(iterator)->it());
  //_objects.erase(std::find(_objects.begin(), _objects.end(), object));
}

template <class T>
bool ObjectList<T>::checkIteratorInList(IteratorBase* iterator) const
{
  const_iterator it;
  for (it = this->_objects.begin(); it != this->_objects.end(); ++it)
    if (static_cast<Iterator*>(iterator)->it() == it)
      return true;
  printf("ObjectList:: checkIteratorInList:: ITERATOR NOT IN THE LIST '%s' (UN-SYNC SHOULD NOT HAPPEN!!)\n", this->name().c_str());
  return false;
}

template <class T>
bool ObjectList<T>::checkObjectInList(BaseObject* obj) const
{
  T* object = dynamic_cast<T*>(obj);
  if (object != NULL)
  {
    printf("EXTREME BUG: Object does not exist anymore!! (or a bug in ObjectList)\n");
    return false;;
  }

  const_iterator it;
  for (it = this->_objects.begin(); it != this->_objects.end(); ++it)
  {
    if (*it == object)
      return true;
  }
  printf("ObjectList:: checkObjectInList:: OBJECT NOT IN THE LIST '%s' (UN-SYNC SHOULD NOT HAPPEN!!)\n", this->name().c_str());
  return false;
}


#endif /* _OBJECT_LIST_H */
