/*!
 * @file base_object.h
 * @brief Definition of the BaseObject class.
 *
 * This is a global handler for all classes Object and Class names
 *
 * BaseObject is the class, that handles object registration and
 * is the only write-access member of ClassList, where the Objects
 * References are stored.
 */


#ifndef __BASE_OBJECT_H_
#define __BASE_OBJECT_H_

#include "object_list.h"
#include "util/sigslot/slot.h"

#include <string>

class TiXmlNode;
class TiXmlElement;
class ClassList;

//! A class all other classes are derived from
class BaseObject : public sigslot::has_slots<>
{
  //! Declare an ObjectList for this Class.
  ObjectListDeclaration(BaseObject);
public:
  BaseObject (const std::string& objectName = "");

  virtual ~BaseObject ();

  virtual void loadParams(const TiXmlElement* root);
  void setName (const std::string& newName);
  /** returns the Name of this Object */
  inline const std::string& getName() const { return this->objectName; };
  /** returns the Name of this Object as a C-compliant string (const char*) */
  inline const char* getCName() const { return this->objectName.c_str(); };
  /** @returns the XML-Element with whicht this Object was loaded */
  inline TiXmlNode* getXmlElem() const { return this->xmlElem; };

  //  /** @returns the className of the corresponding Object */
  //inline const std::string& getClassName() const { return this->className; }
  /** @returns the className of the corresponding Object as a C-compliant string (const char*) */
  inline const char* getClassCName() const { return _classes.front()._objectList->name().c_str(); };
  /** @returns the ClassName of the Topmost Object of the ClassStack */
  inline const std::string& getClassName() const { return _classes.front()._objectList->name(); };

  /** @returns the ClassID of this Object */
  inline const ClassID& getClassID() const { return _classes.front()._objectList->identity(); }
  /** @returns the ID of the Topmost object of the ClassStack. */
  inline const ClassID& getLeafClassID() const { return leafClassID; }

  bool isA(const ObjectListBase& objectList) const;
  bool isA(const ClassID& classID) const;
  bool isA(int classID) const;
  bool isA(const std::string& className) const;

  void listInheritance() const;

  /** @param classID comparer for a ClassID @returns true on match, false otherwise */
  bool operator==(int classID) const  { return this->isA(classID); };
  /** @param objectName: the name to check. * @returns true on match, false otherwise. */
  bool operator==(const std::string& objectName) const { return this->objectName == objectName;};

protected:
  template<class T> void registerObject(T* object, ObjectList<T>& list);

protected:
  std::string        objectName;       //!< The name of this object
  ClassID            leafClassID;      //!< classId of leaf class

private:

  TiXmlNode*         xmlElem;          //!< The XML Element with wich this Object was loaded(saved).

  //////////////////////////////
  //// Type Definition Part ////
  //////////////////////////////
  //! A ClassEntry so we can store Classes inside of Objects
  struct ClassEntry
  {
    /** Simple Constuctor @param objectList the ObjectList, @param iterator the (intrusive) Iterator inside of the ObjectList */
    inline ClassEntry (ObjectListBase* objectList, ObjectListBase::IteratorBase* iterator) : _objectList(objectList), _iterator(iterator) {}
    ObjectListBase*                _objectList;  //!< An ObjectList this Object is part of
    ObjectListBase::IteratorBase*  _iterator;    //!< An iterator pointing to the position of the Object inside of the List.
  };
  typedef std::list<ClassEntry>       ClassEntries;   //!< Type definition for the List.

  std::string                         className;    //!< the name of the class
  ClassEntries                        _classes;     //!< All Classes this object is part of.
};


/**
 * @brief Registeres an Object of Type T to objectList
 * @param object The Object to append to the objectList.
 * @param objectList The ObjectList to append the Object to.
 *
 * This function is essential to integrate objects into their designated ObjectList.
 * Remember if you do not want objects to be stored in Lists (less overhead),
 * do not attempt to call this function.
 */
template<class T>
inline void BaseObject::registerObject(T* object, ObjectList<T>& objectList)
{
  this->_classes.push_front(ClassEntry(&objectList, objectList.registerObject(object)));
  this->leafClassID = objectList.identity();
}

#endif /* __BASE_OBJECT_H_ */
