/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_SHELL

#include "shell_command_class.h"

#include "shell_command.h"

#include "debug.h"
#include "compiler.h"



namespace OrxShell
{
  ObjectListDefinition(ShellCommandClass);

  CmdClassList* ShellCommandClass::_commandClassList = NULL;

  /**
   * @brief creates a new ShellCommandClass
   * @param className the Name of the command-class to create
   */
  ShellCommandClass::ShellCommandClass(const std::string& className)
      : _className(className)
  {
    this->registerObject(this, ShellCommandClass::_objectList);
    this->setName(className);

    if (ShellCommandClass::_commandClassList == NULL)
      ShellCommandClass::_commandClassList = new CmdClassList;
    ShellCommandClass::_commandClassList->push_back(this);
  }

  /**
   * destructs the shellCommandClass again
   */
  ShellCommandClass::~ShellCommandClass()
  {
    while(!this->_commandList.empty())
      delete this->_commandList.back();

    if (ShellCommandClass::_commandClassList != NULL)
    {
      CmdClassList::iterator delClass = std::find(ShellCommandClass::_commandClassList->begin(), ShellCommandClass::_commandClassList->end(), this);
      if (delClass != ShellCommandClass::_commandClassList->end())
        ShellCommandClass::_commandClassList->erase(delClass);
    }
  }

  /**
   * @param command the Command to register.
   */
  void ShellCommandClass::registerCommand(ShellCommand* command)
  {
    this->_commandList.push_back(command);
  }

  /**
   * @brief Unregisters a command.
   * @param command the Command to unregister.
   */
  void ShellCommandClass::unregisterCommand(ShellCommand* command)
  {
    CmdList::iterator delC = std::find(this->_commandList.begin(), this->_commandList.end(), command);
    if (delC != this->_commandList.end())
      this->_commandList.erase(delC);
  }

  /**
   * @brief unregisters all Commands that exist
   */
  void ShellCommandClass::unregisterAllCommands()
  {
    // unregister all commands and Classes
    CmdClassList::iterator classIT;
    if (ShellCommandClass::_commandClassList == NULL)
      return;

    while (!ShellCommandClass::_commandClassList->empty())
      delete ShellCommandClass::_commandClassList->back();
    delete ShellCommandClass::_commandClassList;
    ShellCommandClass::_commandClassList = NULL;
  }


  /**
   * @brief collects the Commands registered to some class.
   * @param className the name of the Class to collect the Commands from.
   * @param stringList a List to paste the Commands into.
   * @returns true on success, false otherwise
   */
  bool ShellCommandClass::getCommandListOfClass(const std::string& className, std::list<std::string>& stringList)
  {
    if (ShellCommandClass::_commandClassList == NULL)
      return false;


    CmdClassList::const_iterator elem;
    for(elem = ShellCommandClass::_commandClassList->begin(); elem != ShellCommandClass::_commandClassList->end(); elem++)
    {
      if (className == (*elem)->getName())
      {
        CmdList::iterator command;
        for(command = (*elem)->_commandList.begin(); command != (*elem)->_commandList.end(); command++)
          stringList.push_back((*command)->getName());
        return true;
      }
    }
    return false;
  }


  /**
   * @brief checks if a Class is already registered to the Commands' class-stack
   * @param className the Name of the Class to check for
   * @returns the CommandClass if found, NULL otherwise
   */
  ShellCommandClass* ShellCommandClass::getCommandClass(const std::string& className)
  {
    if (ShellCommandClass::_commandClassList == NULL)
      return false;


    CmdClassList::const_iterator classIT;
    for (classIT = ShellCommandClass::_commandClassList->begin(); classIT != ShellCommandClass::_commandClassList->end(); classIT++)
      if (className == (*classIT)->_className)
        return (*classIT);
    return NULL;
  }

  /**
   * @brief checks if a Class is already registered to the Commands' class-stack
   * @param className the Name of the Class to check for
   * @returns the CommandClass if found, NULL otherwise
   */
  bool ShellCommandClass::exists(const std::string& className)
  {
    return (ShellCommandClass::getCommandClass(className) != NULL);
  }


  /**
   * @brief searches for a CommandClass
   * @param className the name of the CommandClass
   * @returns the CommandClass if found, or a new CommandClass if not
   */
  ShellCommandClass* ShellCommandClass::acquireCommandClass(const std::string& className)
  {
    ShellCommandClass* cmdClass = ShellCommandClass::getCommandClass(className);
    if (cmdClass != NULL)
      return (cmdClass);
    return new ShellCommandClass(className);
  }

  /**
   * @brief displays help about ShellCommandClass
   * @param className: the Class of Commands to show help about
   */
  void ShellCommandClass::help(const std::string& className)
  {
    if (ShellCommandClass::_commandClassList == NULL)
    {
      PRINT(0)("No Commands Registered\n");
      return;
    }
    if (className.empty())
      PRINT(0)("===== Displaying %d registered Classes:\n", ShellCommandClass::_commandClassList->size());


    CmdClassList::iterator classIT;
    for (classIT = ShellCommandClass::_commandClassList->begin(); classIT != ShellCommandClass::_commandClassList->end(); classIT++)
    {
      if (className.empty() || className == (*classIT)->_className)
      {
        PRINT(0)("Class:'%s' registered %d commands: \n", (*classIT)->_className.c_str(), (*classIT)->_commandList.size());
        CmdList::const_iterator cmdIT;
        for (cmdIT = (*classIT)->_commandList.begin(); cmdIT != (*classIT)->_commandList.end(); cmdIT++)
        {
          PRINT(0)("  command:'%s' : params:%d: ", (*cmdIT)->getCName(), (*cmdIT)->executor->getParamCount());
          /// FIXME
          /*          for (unsigned int i = 0; i< elem->paramCount; i++)
            PRINT(0)("%s ", ShellCommand::paramToString(elem->parameters[i]));*/
          if (!(*cmdIT)->description.empty())
            PRINT(0)("- %s", (*cmdIT)->description.c_str());
          PRINT(0)("\n");
        }
        if (likely(!className.empty()))
          return;
      }
    }
    PRINTF(3)("Class '%s' not found in Command's classes\n", className.c_str());
  }
}




