/*!
 * @file cr_engine.h
 * @brief The collision reaction engine, defining generic collision reactions to collision events
 *
 * some parts of this module are tuned for efficiency. They are probably not self-explenatory anymore :D
 *  - Collision/ CollisionEvent objects recycling: This class contains a class of precached objects of these types
 *      they are used for fast registration of collision events: These objects can be get by the interface functions and
 *      are returned after one cycle automaticly by reseting the cached lists to its initial state. So do not wonder :D
 */

#ifndef _CR_ENGINE_
#define _CR_ENGINE_

#include "base_object.h"

#include <vector>

class WorldEntity;


namespace CoRe
{
  class CollisionHandle;
  class Collision;
  class CollisionEvent;

  //! A default singleton class.
  class CREngine : public BaseObject
  {
    ObjectListDeclaration(CREngine);

  public:
    typedef enum ReactionType {
      CR_PHYSICS_MOMENTUM   = 0,           //!< physical reaction: conservervation of momentum
      CR_PHYSICS_STEP_BACK,                //!< physical reaction: just go to the last position without collisions
      CR_PHYSICS_GROUND_WALK,              //!< physical reaction: stand on the ground, no movement: simple normal force away from the gravity force
      CR_PHYSICS_FULL_WALK,                //!< physical reaction: walking on the ground (inkl. hills etc)
      CR_PHYSICS_DAMAGE,                   //!< physical reaction: daling damage according to the object energy and their structural stability

      CR_OBJECT_DAMAGE,                    //!< object raction: deals damage according to the objects specific damage potential (like weapons, nukes, etc.)
      CR_OBJECT_PICKUP,                    //!< object rection: calling the objects pickup functions, let them handle the collision (once!)

      CR_VERTEX_TRAFO,                     //!< vertex trafo: transforming the vertex according to the damage

      CR_SPECIAL_CALLBACK,                 //!< special: call a callback function

      CR_NUMBER
    };

    typedef enum CollisionType {
      CR_COLLISION_TYPE_AXIS_X       = 0,  //!< collision on x axis
      CR_COLLISION_TYPE_AXIS_X_NEG,        //!< collision on negative x axis
      CR_COLLISION_TYPE_AXIS_Y,            //!< collision on y axis
      CR_COLLISION_TYPE_AXIS_Y_NEG,        //!< collision on negative y axis
      CR_COLLISION_TYPE_AXIS_Z,            //!< collision on z axis
      CR_COLLISION_TYPE_AXIS_Z_NEG,        //!< collision on negative z axis
      CR_COLLISION_TYPE_OBB,               //!< object aligned bounding box collide

      CR_COLLISION_TYPE_NUMBER
    };


#define COLLISION_TYPE_AXIS_X      1
#define COLLISION_TYPE_AXIS_X_NEG  2
//!< the collision axis y collision event
#define COLLISION_TYPE_AXIS_Y      3
#define COLLISION_TYPE_AXIS_Y_NEG  4
//!< the collision axis z collision event
#define COLLISION_TYPE_AXIS_Z      5
#define COLLISION_TYPE_AXIS_Z_NEG  6
//!< the collision is a obb collision
#define COLLISION_TYPE_OBB         8

    virtual ~CREngine(void);

    /** @returns a Pointer to the only object of this Class */
    inline static CREngine* getInstance() { if (!singletonRef) singletonRef = new CREngine();  return singletonRef; };

    CollisionHandle* subscribeReaction(WorldEntity* worldEntity, ReactionType type);
    bool unsubscribeReaction(CollisionHandle* collisionHandle);


    Collision* popCollisionObject();
    CollisionEvent* popCollisionEventObject();
    void reset();

    void handleCollisions();

    void debug();


  private:
    CREngine();
    void init();

    void flushCollisions();


  private:
    std::vector<CollisionHandle*>       collisionHandles;         //!< list with the collision handles

    std::vector<Collision*>             collisionsUsed;           //!< a list of used, cached collisions
    std::vector<Collision*>             collisionsUnused;         //!< a list of unused, cached collisions

    std::vector<CollisionEvent*>        collisionEventsUsed;      //!< a list of used, cached collision events
    std::vector<CollisionEvent*>        collisionEventsUnused;    //!< a list of unused, cached collision events

    static CREngine*                    singletonRef;             //!< the reference to the CREngine object (singleton)
  };

}
#endif /* _CR_ENGINE_ */
