/*!
 * @file collision_tube.h
 *
 * collision tube collects all collisions from all entities
 *
 *  The collision events are saved in the _collisionList vector in a fixed hirarchy: a collision is defined by the world entities
 *  that collide. For each collision there is an entry in _collisionList. Each collision itself again contains collision events which are
 *  defined as collisions between the boundibg volumes (BV) of the world entities. This could look like this:
 *
 *  - Collision (WorldEntity_i  <=> WorldEntity_j)
 *     +- CollisionEvent( some BV <=> some other BV)
 *     +- CollisionEvent( some BV <=> some other BV)
 *  - Collision (WorldEntity_k <=> WorldEntity_l)
 *     +- CollisionEvent( some BV <=> some other BV)
 *     +- CollisionEvent( some BV <=> some other BV)
 *     +- CollisionEvent( some BV <=> some other BV)
 *     +- CollisionEvent( some BV <=> some other BV)
 *  - ... etc ...
 *
 *
 *  When the collisions are processed by the handleCollision() function each collision pair is checked for their reactions (since each
 *  WorldEntity can define several reactions to a collision). After all the reactions are calculated and applied the collision object is
 *  put back.
 */

#ifndef _COLLISION_TUBE_H
#define _COLLISION_TUBE_H

#include "base_object.h"
#include "cr_engine.h"
#include "world_entity.h"

#include "vector.h"
#include <vector>

class Collision;
class WorldEntity;
class BoundingVolume;

namespace CoRe
{

  class CollisionReaction;

  //! A class containing all CollisionEvents (structured as defined in the file doxygen tags)
  class CollisionTube : public BaseObject
  {
    ObjectListDeclaration(CollisionTube);


  public:
    CollisionTube();
    virtual ~CollisionTube();

    /** @returns true if at least one of both WorldEntities are subscribed for a collision check */
    inline bool isReactive(const WorldEntity& entityA, const WorldEntity& entityB) const
    { return (entityA.isReactive() && entityB.isReactive()); }

    void registerCollisionEvent(WorldEntity* entityA, WorldEntity* entityB, BoundingVolume* bvA, BoundingVolume* bvB);
    void registerCollisionEvent(int type, WorldEntity* entity, WorldEntity* groundEntity,
                                const Vector& normal, const Vector& position, bool bInWall = false);

    void handleCollisions();


  private:
    /* private copy constructor so this object can't be passed as a */
    CollisionTube(const CollisionTube& tube) {}


  private:
    std::vector<Collision*>        _collisionList;      //!< the list of collisions since the last processing

  };

}

#endif /* _COLLISION_TUBE_H */
