/*!
 * @file collision_handle.h
 * @brief Definition of a collision handle: used for accessing per world entity collision events and reactions
*/

#ifndef _COLLISION_HANDLE_H
#define _COLLISION_HANDLE_H

#include "base_object.h"
#include "cr_engine.h"

#include <vector>
#include <list>


class Collision;
class WorldEntity;

namespace CoRe
{

  class CollisionReaction;



  //! A class for defining collision reactions and storing events
  class CollisionHandle : public BaseObject
  {
    ObjectListDeclaration(CollisionHandle);

  public:
    CollisionHandle(WorldEntity* owner, CREngine::ReactionType type);
    virtual ~CollisionHandle();

    void reset();

    void addTarget(const ClassID& target);
    Collision* registerCollision(WorldEntity* entityA, WorldEntity* entityB);
    void registerSharedCollision(Collision* collision);
    void registerCollisionEvent(CollisionEvent* collisionEvent);

    /** @returns true if regiestered some new collision events in this tick frame */
    inline bool isCollided() const { return this->bCollided; }
    /** @returns true if this collision handle has already been dispatched */
    inline bool isDispatched() const { return this->bDispatched; }
    /** @returns true if this handle should be pulled also if there are no collisions */
    inline bool isContinuousPoll() const { return this->bContinuousPoll; }
    /** @returns the type */
    inline CREngine::ReactionType getType() const { return this->type; }

    void handleCollisions();


  private:
    void flushCollisions();
    bool filterCollisionEvent(CollisionEvent* collisionEvent);
    bool filterCollision(Collision* collision);



  private:
    WorldEntity*                  owner;                   //!< the worldenity this reaction will be applied on
    CREngine::ReactionType              type;                    //!< the reaction type

    bool                          bContinuousPoll;         //!< if this is true
    bool                          bDispatched;             //!< true if this handle has already been dispatched
    bool                          bStopOnFirstCollision;   //!< true if the cd of this object should be terminated after one match
    bool                          bCollided;               //!< true if the CollsionHandle has registered some new collisions

    std::vector<Collision*>       collisionList;           //!< a list full of collisions
    std::vector<ClassID>          targetList;              //!< a list of target classes for filtering @TODO TAKE SET INSTEAD OF VECTOR HERE

    CollisionReaction*            collisionReaction;       //!< reference to the collision reaction object

  };

}
#endif /* _COLLISION_HANDLE_H */
