/*!
 * @file ini_parser.h
 * A small ini file parser
 *
 * Can be used to find a defined [Section] in an ini file and get the VarName = Value entries
 */
#ifndef _INI_PARSER_H
#define _INI_PARSER_H

#include <string>
#include <list>

//! ini-file parser
/**
 * This class can be used to load an initializer file and parse it's contents for variablename=value pairs.
 */
class IniParser
{
public:
  ////////////////////////////////////
  /// A class for a Ini-Node. The base of all INI-elements.
  class Node
  {
  public:
    Node(const std::string& name, const std::string& comment = "");
    //! Simple destructor
    virtual ~Node() {};
    /** @returns the name of the Node */
    const std::string& name() const  { return _name; };
    /** @returns the Comment of the Node */
    const std::string& comment() const  { return _comment; };
    /** @param name the name to set for this node */
    void setName(const std::string& name) { this->_name = name; };
    /** @param comment the Comment to set for this node */
    void setComment(const std::string& comment) { this->_comment = comment; };

    /** @param name the name to compare against this nodes name @returns true on match */
    bool operator==(const std::string& name) const { return _name == name; };

    /** @brief displays some debug information about the node */
    virtual void debug() const = 0;

  private:
    std::string         _comment;  //!< A Comment that is appendet to the Top of this Node
    std::string         _name;     //!< name of a given Node
  };

  //! a class for Entries in the Parser's File's Sections
class Entry : public Node
  {
  public:
    Entry(const std::string& name, const std::string& value = "", const std::string& comment = "");
    /** @returns the Value of the Entry */
    const std::string& value() const { return _value; };
    /** @param value sets the value of the Entry */
    void setValue (const std::string& value) { _value = value; };

    virtual void debug() const;

  public:
    typedef std::list<Entry>       list;           //!< A Type definition for lists of Entries.
    typedef list::iterator         iterator;       //!< A Type definition for iterators of Entries.
    typedef list::const_iterator   const_iterator; //!< A Type definition for constant iterators of Entries.

  private:
    std::string                    _value;         //!< value of a given Entry
  };

  //! a clas for Sections in the Parser's file
class Section : public Node
  {
  public:
    Section(const std::string& sectionName, const std::string& comment = "");

    Entry& addEntry(const std::string& entryName, const std::string& value = "", const std::string& comment = "");
    bool editEntry(const std::string& entryName, const std::string& value, bool createMissing = true);
    const std::string& getValue(const std::string& entryName, const std::string& defaultValue = "") const;

    bool setEntryComment(const std::string& entryName, const std::string& comment);
    const std::string& getEntryComment(const std::string& entryName) const;

    /** @returns the List of Entries */
    const Entry::list& entries() const { return _entries; }
    Entry* getEntry(const std::string& entryName);

    Entry::const_iterator getEntryIt(const std::string& entryName) const;
    /** @returns an Iterator pointing to the beginning of the entries. */
    Entry::iterator begin() { return _entries.begin(); };
    /** @returns a constant Iterator pointing to the beginning of the entries */
    Entry::const_iterator begin() const { return _entries.begin(); };
    /** @returns an Iterator pointing to the end of the entries */
    Entry::iterator end() { return _entries.end(); };
    /** @returns a constant Iterator pointing to the end of the entries */
    Entry::const_iterator end() const { return _entries.end(); };

    void clear();

    virtual void debug() const;

  public:
    typedef std::list<Section>      list;            //!< A Type definition for lists of Sections
    typedef list::iterator          iterator;        //!< A Type definition for iterators of Sectionlists.
    typedef list::const_iterator    const_iterator;  //!< A Type definition for constant iterators of Sectionlists.

  private:
    Entry::list                     _entries;        //!< a list of entries for this section
  };

  //! A class for a INI-file.
class Document : public Node
  {
  public:
    Document(const std::string& fileName, const std::string& comment = "");

    Section& addSection(const std::string& sectionName, const std::string& comment = "");
    bool removeSection(const std::string& sectionName);
    bool setSectionsComment(const std::string& sectionName, const std::string& comment);
    const std::string& getSectionsComment(const std::string& sectionName) const;

    /** @returns list of all sections */
    const Section::list& sections() const { return _sections; }
    Section* getSection(const std::string& sectionName);

    Section::const_iterator getSectionIt(const std::string& sectionName) const;
    /** @returns an Iterator poining to the beginning of the Sections List */
    Section::iterator begin() { return _sections.begin(); };
    /** @returns a constant Iterator poining to the beginning of the Sections List */
    Section::const_iterator begin() const { return _sections.begin(); };
    /** @returns an Iterator poining to the end of the Sections List */
    Section::iterator end() { return _sections.end(); };
    /** @returns a constant Iterator poining to the end of the Sections List */
    Section::const_iterator end() const { return _sections.end(); };

    bool addEntry(const std::string& sectionName, const std::string& entryName, const std::string& value = "", const std::string& comment = "");
    bool editEntry(const std::string& sectionName, const std::string& entryName, const std::string& value, bool createMissing = true);
    const std::string& getValue(const std::string& sectionName, const std::string& entryName, const std::string& defaultValue = "") const;

    bool setEntryComment(const std::string& sectionName, const std::string& entryName, const std::string& comment);
    const std::string& getEntryComment(const std::string& sectionName, const std::string& entryName) const;

    void clear();

    virtual void debug() const;

  private:
    Section::list       _sections;        //!< a list of all stored Sections of the Parser.
  };

public:
  IniParser (const std::string& filename = "");
  virtual ~IniParser ();

  /** @returns true if the file is opened, false otherwise */
  bool isOpen() const { return !this->_document.sections().empty(); };
  /** @returns the fileName we have opened. */
  const std::string& getFileName() const { return this->_document.name(); };

  /// Read and Write the File
  bool readFile(const std::string& fileName, bool keepSettings = false);
  bool writeFile(const std::string& fileName = "") const;

  //! @param fileComment the comment of the Document */
  void setFileComment(const std::string& fileComment) { this->_document.setComment(fileComment); };
  /** @returns comments for the File. */
  const std::string& getFileComment() const { return this->_document.comment(); };

  /// Working with Sections.
  Section& addSection(const std::string& sectionName);
  // iterate through sections with these Functions
  //! @see Section::getSection()
  Section* getSection(const std::string& sectionName) { return this->_document.getSection(sectionName); };
  //! @see Document::getSectionIt()
  Section::const_iterator getSectionIt(const std::string& sectionName) const { return this->_document.getSectionIt(sectionName); };

  //! @see Document::begin()
  Section::iterator begin() { return this->_document.begin(); };
  //! @see Document::begin()
  Section::const_iterator begin() const { return this->_document.begin(); };
  //! @see Document::end()
  Section::iterator end() { return this->_document.end(); };
  //! @see Document::end()
  Section::const_iterator end() const { return this->_document.end(); };
  //! @see Document::setSectionComment()
  void setSectionsComment(const std::string& sectionName, const std::string& comment) { this->_document.setSectionsComment(sectionName, comment); };
  //! @see Document::getSectionsComment()
  const std::string& getSectionsComment(const std::string& sectionName) const { return this->_document.getSectionsComment(sectionName); };

  /// Working on Entries. (globally)
  //! @see Document::addEntry()
  bool addEntry(const std::string& sectionName, const std::string& entryName, const std::string& value, const std::string& comment)
  { return this->_document.addEntry(sectionName, entryName, value, comment); };
  //! @see Document::getValue()
  const std::string& getValue(const std::string& sectionName, const std::string& entryName, const std::string& defaultValue = "") const
  { return this->_document.getValue(sectionName, entryName, defaultValue); };
  //! @see Document::editEntry()
  bool editEntry(const std::string& sectionName, const std::string& entryName, const std::string& value, bool createMissing = true)
  { return this->_document.editEntry(sectionName, entryName, value, createMissing); };
  //! @see Document::setEntryComment()
  void setEntryComment(const std::string& sectionName, const std::string& entryName, const std::string& comment)
  { this->_document.setEntryComment(sectionName, entryName, comment); };
  //! @see Document::getEntryComment()
  const std::string& getEntryComment(const std::string& sectionName, const std::string& entryName) const
  { return this->_document.getEntryComment(sectionName, entryName); };

  // maintenance.
  void debug() const;

private:
  void setNodeComment(Node* node, std::list<std::string>* comments);
private:
  std::string                      _fileName;        //!< The name of the File that is parsed here.
  Document                         _document;        //!< The Document inside of the file.

  static const std::string         _emptyString;     //!< Just an Empty String that will be returned if nothing else is found.
};

#endif /* _INI_PARSER_H */
