/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Stefan Lienard
   co-programmer: ...
*/

#include "mapped_water.h"
#include "util/loading/load_param.h"
#include "util/loading/factory.h"
#include "state.h"
#include "t_animation.h"
#include <cmath>
#include "glgui.h"
#include "shell_command.h"
#include "script_class.h"

#include "resource_shader.h"

#include "class_id_DEPRECATED.h"
ObjectListDefinitionID(MappedWater, CL_MAPPED_WATER);
CREATE_FACTORY(MappedWater);

SHELL_COMMAND(gui, MappedWater, toggleGui);
SHELL_COMMAND(output, MappedWater, saveParams);

CREATE_SCRIPTABLE_CLASS(MappedWater,
                        addMethod("waterUV", Executor2<MappedWater, lua_State*, float, float>(&MappedWater::fadeWaterUV))
                        ->addMethod("waterFlow", Executor2<MappedWater, lua_State*, float, float>(&MappedWater::fadeWaterFlow))
                        ->addMethod("shineSize", Executor2<MappedWater, lua_State*, float, float>(&MappedWater::fadeShineSize))
                        ->addMethod("shineStrength", Executor2<MappedWater, lua_State*, float, float>(&MappedWater::fadeShineStrength))
                        ->addMethod("reflStrength", Executor2<MappedWater, lua_State*, float, float>(&MappedWater::fadeReflStrength))
                        ->addMethod("refraction", Executor2<MappedWater, lua_State*, float, float>(&MappedWater::fadeRefraction))
                        ->addMethod("waterHeight", Executor2<MappedWater, lua_State*, float, float>(&MappedWater::fadeWaterHeight))
                        ->addMethod("waterColor", Executor4<MappedWater, lua_State*, float, float, float, float>(&MappedWater::fadeWaterColor)));


/**
 * @brief constructor
 * @param root xml data
 */
MappedWater::MappedWater(const TiXmlElement* root)
{
  this->registerObject(this, MappedWater::_objectList);
  this->toList(OM_ENVIRON);

  /// sets start values and parameters
  this->initParams();
  // now the standard values were loaded, if the values are specified in the .oxw file
  // loadParams will overwrite the standard values with the specified values
  if (root != NULL)
    this->loadParams(root);

  /// initialization of the textures
  this->initTextures();

  /// initialization of the shaders
  this->initShaders();

  /// calculation of the 4 verts of the water quad
  this->calcVerts();

  // init gui
  this->box = NULL;
}

/**
 * @brief deltes shader and the uniform used by the camera
 */
MappedWater::~MappedWater()
{
  delete cam_uni;
  delete color_uni;
  delete light_uni;
  delete shineSize_uni;
  delete shineStrength_uni;
  delete reflStrength_uni;
  delete refr_uni;
}

/**
 * @brief initialization of loadable parameters, sets start standard values
 */
void MappedWater::initParams()
{
  // those standardvalues will be overwritten if they're also set in the oxw file
  this->setWaterPos(0, 0, 0);
  this->setWaterSize(100, 100);
  this->setWaterUV(9);
  this->setWaterFlow(0.08f);
  this->setLightPos(0, 10, 0);
  this->setWaterAngle(0);
  this->setNormalMapScale(0.25f);
  this->setWaterColor(0.1f, 0.2f, 0.4f);
  this->setShineSize(128);
  this->setShineStrength(0.7f);
  this->setReflStrength(1.0f);
  this->setRefraction(0.009f);

  // initialization of the texture coords, speeds etc...
  // normalUV wont change anymore
  this->normalUV = this->waterUV * this->kNormalMapScale;
  // move and move2 are used for the reflection and refraction, the values are changed in tick()
  this->move = 0;
  this->move2 = this->move * this->kNormalMapScale;

  // initalize fading bools
  this->bFadeWaterHeight = false;
  this->bFadeWaterUV = false;
  this->bFadeWaterFlow = false;
  this->bFadeShineSize = false;
  this->bFadeShineStrength = false;
  this->bFadeReflStrength = false;
  this->bFadeRefraction = false;
  this->bFadeWaterColor = false;
}

/**
 * @brief initialization of the textures
 */
void MappedWater::initTextures()
{
  // sets parameters for the textures
  this->textureSize = 512;
  unsigned int channels = 32;
  GLenum type = GL_RGBA;

  // set up refleciton texture
  Texture reflTex(GL_TEXTURE_2D, this->textureSize, this->textureSize, channels, type);
  mat.setDiffuseMap(reflTex, 0);
  // set up refraction texture
  Texture refrTex(GL_TEXTURE_2D, this->textureSize, this->textureSize, channels, type);
  mat.setDiffuseMap(refrTex, 1);
  // load normal map
  mat.setDiffuseMap("pictures/water_normalmap.bmp", GL_TEXTURE_2D, 2);
  // load dudv map
  mat.setDiffuseMap("pictures/water_dudvmap.bmp", GL_TEXTURE_2D, 3);

  // sets texture parameters for reflection texture
  glBindTexture(GL_TEXTURE_2D, this->mat.diffuseTextureID(0));
  glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
  glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
  // sets texture parameters for refraction texture
  glBindTexture(GL_TEXTURE_2D, this->mat.diffuseTextureID(1));
  glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
  glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
}

/**
 * @brief initialization of the shaders
 */
void MappedWater::initShaders()
{
  // load shader files
  shader = ResourceShader( "shaders/mapped_water.vert", "shaders/mapped_water.frag");

  this->shader.activateShader();
  // Set the variable "reflection" to correspond to the first texture unit
  Shader::Uniform(shader, "reflection").set(0);
  // Set the variable "refraction" to correspond to the second texture unit
  Shader::Uniform(shader, "refraction").set(1);
  // Set the variable "normalMap" to correspond to the third texture unit
  Shader::Uniform(shader, "normalMap").set(2);
  // Set the variable "dudvMap" to correspond to the fourth texture unit
  Shader::Uniform(shader, "dudvMap").set(3);
  // Set the variable "depthMap" to correspond to the fifth texture unit
  Shader::Uniform(shader, "depthMap").set(1);
  // Give the variable "waterColor" a blue color
  color_uni = new Shader::Uniform(shader, "waterColor");
  color_uni->set(waterColor.x, waterColor.y, waterColor.z, 1.0f);
  // Give the variable "lightPos" our hard coded light position
  light_uni = new Shader::Uniform(shader, "lightPos");
  light_uni->set(lightPos.x, lightPos.y, lightPos.z, 1.0f);
  // Set the variable "shine"
  shineSize_uni = new Shader::Uniform(shader, "kShine");
  shineSize_uni->set(this->shineSize);
  // Set the variable "shineStrength"
  shineStrength_uni = new Shader::Uniform(shader, "shineStrength");
  shineStrength_uni->set(this->shineStrength);
  // Set the variable "reflStrength"
  reflStrength_uni = new Shader::Uniform(shader, "reflStrength");
  reflStrength_uni->set(this->reflStrength);
  // Set the variable "refraction"
  refr_uni = new Shader::Uniform(shader, "kRefraction");
  refr_uni->set(this->refraction);
  // uniform for the camera position
  cam_uni = new Shader::Uniform(shader, "cameraPos");

  this->shader.deactivateShader();
}

/**
 * @brief calculates the 4 verts of the water quad
 */
void MappedWater::calcVerts()
{
  float deg2radtemp = this->waterAngle / 180 * PI;

  this->waterVerts[2] = this->waterVerts[0] + cos(deg2radtemp) * this->xWidth;
  this->waterVerts[3] = this->waterVerts[1] + sin(deg2radtemp) * this->xWidth;
  this->waterVerts[4] = this->waterVerts[0] + cos(deg2radtemp) * this->xWidth - sin(deg2radtemp) * this->zWidth;
  this->waterVerts[5] = this->waterVerts[1] + sin(deg2radtemp) * this->xWidth + cos(deg2radtemp) * this->zWidth;
  this->waterVerts[6] = this->waterVerts[0] - sin(deg2radtemp) * this->zWidth;
  this->waterVerts[7] = this->waterVerts[1] + cos(deg2radtemp) * this->zWidth;
}

/**
 * @brief resets the waterColor in the Shader
 * @param r new value for red
 * @param g new value for green
 * @param b new value for blue
 */
void MappedWater::resetWaterColor(float r, float g, float b)
{
  this->shader.activateShader();
  this->waterColor = Vector(r, g, b);

  // Give the variable "waterColor" a color
  color_uni->set(waterColor.x, waterColor.y, waterColor.z, 1.0f);

  this->shader.deactivateShader();
}

/**
 * @brief resets the shininess in the Shader
 * @param shine new value for the shininess
 */
void MappedWater::resetShineSize(float shine)
{
  this->shader.activateShader();
  this->shineSize = shine;

  // Set the variable "shine"
  shineSize_uni->set(this->shineSize);

  this->shader.deactivateShader();
}

/**
 * @brief resets the strength of the specular reflection in the Shader
 * @param strength new value for the strength of the specular reflection
 */
void MappedWater::resetShineStrength(float strength)
{
  this->shader.activateShader();
  this->shineStrength = strength;

  // Set the variable "shine"
  shineStrength_uni->set(this->shineStrength);

  this->shader.deactivateShader();
}

/**
 * @brief resets the strength of the reflection in the Shader
 * @param strength new value for the strength of the reflection
 */
void MappedWater::resetReflStrength(float strength)
{
  this->shader.activateShader();
  this->reflStrength = strength;

  // Set the variable "shine"
  reflStrength_uni->set(this->reflStrength);

  this->shader.deactivateShader();
}

/**
 * @brief resets the refraction in the Shader
 * @param refraction new value for the refraction
 */
void MappedWater::resetRefraction(float refraction)
{
  this->shader.activateShader();
  this->refraction = refraction;

  // Set the variable "shine"
  refr_uni->set(this->refraction);

  this->shader.deactivateShader();
}

/**
 * @brief resets the lightPos in the Shader
 * @param x new x value
 * @param y new y value
 * @param z new z value
 */
void MappedWater::resetLightPos(float x, float y, float z)
{
  this->shader.activateShader();
  this->lightPos = Vector(x, y, z);

  // Give the variable "lightPos" our hard coded light position
  light_uni->set(lightPos.x, lightPos.y, lightPos.z, 1.0f);

  this->shader.deactivateShader();
}

/**
 * @brief ends the refraction and saves the graphic buffer into a texture
 * @param root xml data
 */
void MappedWater::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "waterpos", this, MappedWater, setWaterPos);
  LoadParam(root, "watersize", this, MappedWater, setWaterSize);
  LoadParam(root, "wateruv", this, MappedWater, setWaterUV);
  LoadParam(root, "waterflow", this, MappedWater, setWaterFlow);
  LoadParam(root, "lightpos", this, MappedWater, setLightPos);
  LoadParam(root, "waterangle", this, MappedWater, setWaterAngle);
  LoadParam(root, "normalmapscale", this, MappedWater, setNormalMapScale);
  LoadParam(root, "shinesize", this, MappedWater, setShineSize);
  LoadParam(root, "shinestrength", this, MappedWater, setShineStrength);
  LoadParam(root, "reflstrength", this, MappedWater, setReflStrength);
  LoadParam(root, "refraction", this, MappedWater, setRefraction);
  LoadParam(root, "watercolor", this, MappedWater, setWaterColor);
}

/**
 * @brief prints the xml code of the water params
 */
void MappedWater::saveParams()
{
  // it's not too nice, but it works fine
  PRINT(0)("\nMappedWater XML Code:\n<MappedWater>\n");

  PRINT(0)("  <waterpos>%f, %f, %f</waterpos>\n", this->waterVerts[0], this->waterHeight, this->waterVerts[1]);
  PRINT(0)("  <watersize>%f, %f</watersize>\n", this->xWidth, this->zWidth);
  PRINT(0)("  <wateruv>%f</wateruv>\n", this->waterUV);
  PRINT(0)("  <waterflow>%f</waterflow>\n", this->waterFlow);
  PRINT(0)("  <lightpos>%f, %f, %f</lightpos>\n", this->lightPos.x, this->lightPos.y, this->lightPos.z);
  PRINT(0)("  <waterangle>%f</waterangle>\n", this->waterAngle);
  PRINT(0)("  <normalmapscale>%f</normalmapscale>\n", this->kNormalMapScale);
  PRINT(0)("  <shinesize>%f</shinesize>\n", this->shineSize);
  PRINT(0)("  <shinestrength>%f</shinestrength>\n", this->shineStrength);
  PRINT(0)("  <reflstrength>%f</reflstrength>\n", this->reflStrength);
  PRINT(0)("  <refraction>%f</refraction>\n", this->refraction);
  PRINT(0)("  <watercolor>%f, %f, %f</watercolor>\n", this->waterColor.x, this->waterColor.y, this->waterColor.z);

  PRINT(0)("</MappedWater>\n");
}

/**
 * @brief starts the slider gui that lets you edit all water parameters
 */
void MappedWater::toggleGui()
{
  if (this->box == NULL)
  {
    this->box = new OrxGui::GLGuiBox(OrxGui::Vertical);
    {
      OrxGui::GLGuiBox* waterColorBox = new OrxGui::GLGuiBox(OrxGui::Horizontal);
      {
        OrxGui::GLGuiText* waterColorText = new OrxGui::GLGuiText();
        waterColorText->setText("WaterColor");
        waterColorBox->pack(waterColorText);

        OrxGui::GLGuiSlider* waterColorR = new OrxGui::GLGuiSlider();
        waterColorR->setRange(0, 1.0f);
        waterColorR->setValue(this->waterColor.x);
        waterColorR->setStep(0.1f);
        waterColorR->valueChanged.connect(this, &MappedWater::resetWaterColorR);
        waterColorBox->pack(waterColorR);

        OrxGui::GLGuiSlider* waterColorG = new OrxGui::GLGuiSlider();
        waterColorG->setRange(0, 1.0f);
        waterColorG->setStep(0.1f);
        waterColorG->setValue(this->waterColor.y);
        waterColorG->valueChanged.connect(this, &MappedWater::resetWaterColorG);
        waterColorBox->pack(waterColorG);

        OrxGui::GLGuiSlider* waterColorB = new OrxGui::GLGuiSlider();
        waterColorB->setRange(0, 1.0f);
        waterColorB->setStep(0.1f);
        waterColorB->setValue(this->waterColor.z);
        waterColorB->valueChanged.connect(this, &MappedWater::resetWaterColorB);
        waterColorBox->pack(waterColorB);
      }
      this->box->pack(waterColorBox);

      OrxGui::GLGuiBox* waterUVBox = new OrxGui::GLGuiBox(OrxGui::Horizontal);
      {
        OrxGui::GLGuiText* waterUVText = new OrxGui::GLGuiText();
        waterUVText->setText("WaterUV");
        waterUVBox->pack(waterUVText);

        OrxGui::GLGuiSlider* waterUV = new OrxGui::GLGuiSlider();
        waterUV->setRange(1, 20);
        waterUV->setValue(this->waterUV);
        waterUV->setStep(1);
        waterUV->valueChanged.connect(this, &MappedWater::setWaterUV);
        waterUVBox->pack(waterUV);
      }
      this->box->pack(waterUVBox);

      OrxGui::GLGuiBox* waterFlowBox = new OrxGui::GLGuiBox(OrxGui::Horizontal);
      {
        OrxGui::GLGuiText* waterFlowText = new OrxGui::GLGuiText();
        waterFlowText->setText("WaterFlow");
        waterFlowBox->pack(waterFlowText);

        OrxGui::GLGuiSlider* waterFlow = new OrxGui::GLGuiSlider();
        waterFlow->setRange(0.01f, 2);
        waterFlow->setValue(this->waterFlow);
        waterFlow->setStep(0.02f);
        waterFlow->valueChanged.connect(this, &MappedWater::setWaterFlow);
        waterFlowBox->pack(waterFlow);
      }
      this->box->pack(waterFlowBox);

      OrxGui::GLGuiBox* shineSizeBox = new OrxGui::GLGuiBox(OrxGui::Horizontal);
      {
        OrxGui::GLGuiText* shineSizeText = new OrxGui::GLGuiText();
        shineSizeText->setText("ShineSize");
        shineSizeBox->pack(shineSizeText);

        OrxGui::GLGuiSlider* shineSize = new OrxGui::GLGuiSlider();
        shineSize->setRange(1, 128);
        shineSize->setValue(this->shineSize);
        shineSize->setStep(1);
        shineSize->valueChanged.connect(this, &MappedWater::resetShineSize);
        shineSizeBox->pack(shineSize);
      }
      this->box->pack(shineSizeBox);

      OrxGui::GLGuiBox* shineStrengthBox = new OrxGui::GLGuiBox(OrxGui::Horizontal);
      {
        OrxGui::GLGuiText* shineStrengthText = new OrxGui::GLGuiText();
        shineStrengthText->setText("ShineStrength");
        shineStrengthBox->pack(shineStrengthText);

        OrxGui::GLGuiSlider* shineStrength = new OrxGui::GLGuiSlider();
        shineStrength->setRange(0, 1);
        shineStrength->setValue(this->shineStrength);
        shineStrength->setStep(0.1f);
        shineStrength->valueChanged.connect(this, &MappedWater::resetShineStrength);
        shineStrengthBox->pack(shineStrength);
      }
      this->box->pack(shineStrengthBox);

      OrxGui::GLGuiBox* reflStrengthBox = new OrxGui::GLGuiBox(OrxGui::Horizontal);
      {
        OrxGui::GLGuiText* reflStrengthText = new OrxGui::GLGuiText();
        reflStrengthText->setText("ReflStrength");
        reflStrengthBox->pack(reflStrengthText);

        OrxGui::GLGuiSlider* reflStrength = new OrxGui::GLGuiSlider();
        reflStrength->setRange(0, 1);
        reflStrength->setValue(this->reflStrength);
        reflStrength->setStep(0.1f);
        reflStrength->valueChanged.connect(this, &MappedWater::resetReflStrength);
        reflStrengthBox->pack(reflStrength);
      }
      this->box->pack(reflStrengthBox);

      OrxGui::GLGuiBox* refractionBox = new OrxGui::GLGuiBox(OrxGui::Horizontal);
      {
        OrxGui::GLGuiText* refractionText = new OrxGui::GLGuiText();
        refractionText->setText("Refraction");
        refractionBox->pack(refractionText);

        OrxGui::GLGuiSlider* refraction = new OrxGui::GLGuiSlider();
        refraction->setRange(0.001f, 0.1f);
        refraction->setValue(this->refraction);
        refraction->setStep(0.004f);
        refraction->valueChanged.connect(this, &MappedWater::resetRefraction);
        refractionBox->pack(refraction);
      }
      this->box->pack(refractionBox);

      OrxGui::GLGuiBox* lightPosBox = new OrxGui::GLGuiBox(OrxGui::Horizontal);
      {
        OrxGui::GLGuiText* lightPosText = new OrxGui::GLGuiText();
        lightPosText->setText("LightPos");
        lightPosBox->pack(lightPosText);

        OrxGui::GLGuiSlider* lightPosX = new OrxGui::GLGuiSlider();
        lightPosX->setRange(-4000, 4000);
        lightPosX->setValue(this->lightPos.x);
        lightPosX->setStep(15);
        lightPosX->valueChanged.connect(this, &MappedWater::resetLightPosX);
        lightPosBox->pack(lightPosX);

        OrxGui::GLGuiSlider* lightPosY = new OrxGui::GLGuiSlider();
        lightPosY->setRange(-4000, 4000);
        lightPosY->setStep(15);
        lightPosY->setValue(this->lightPos.y);
        lightPosY->valueChanged.connect(this, &MappedWater::resetLightPosY);
        lightPosBox->pack(lightPosY);

        OrxGui::GLGuiSlider* lightPosZ = new OrxGui::GLGuiSlider();
        lightPosZ->setRange(-4000, 4000);
        lightPosZ->setStep(15);
        lightPosZ->setValue(this->lightPos.z);
        lightPosZ->valueChanged.connect(this, &MappedWater::resetLightPosZ);
        lightPosBox->pack(lightPosZ);
      }
      this->box->pack(lightPosBox);

      OrxGui::GLGuiBox* waterHeightBox = new OrxGui::GLGuiBox(OrxGui::Horizontal);
      {
        OrxGui::GLGuiText* waterHeightText = new OrxGui::GLGuiText();
        waterHeightText->setText("WaterHeight");
        waterHeightBox->pack(waterHeightText);

        OrxGui::GLGuiSlider* waterHeight = new OrxGui::GLGuiSlider();
        waterHeight->setRange(100, 370);
        waterHeight->setValue(this->waterHeight);
        waterHeight->setStep(4);
        waterHeight->valueChanged.connect(this, &MappedWater::setWaterHeight);
        waterHeightBox->pack(waterHeight);
      }
      this->box->pack(waterHeightBox);
    }

    this->box->showAll();
    this->box->setAbsCoor2D(300, 40);
    OrxGui::GLGuiHandler::getInstance()->activate();
    OrxGui::GLGuiHandler::getInstance()->activateCursor();
  }
  else
  {
    OrxGui::GLGuiHandler::getInstance()->deactivate();
    OrxGui::GLGuiHandler::getInstance()->deactivateCursor();
    delete this->box;
    this->box = NULL;
  }
}

/**
 * @brief activates the water shader and draws a quad with four textures on it
 */
void MappedWater::draw() const
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  // don't use glRotate or glTranslate here... the shader won't work anymore

  mat.select();

  this->shader.activateShader();

  // reset the camera uniform to the current cam position
  Vector pos = State::getCameraNode()->getAbsCoor();
  cam_uni->set(pos.x, pos.y, pos.z, 1.0f);

  glDisable(GL_BLEND);

  // TODO change texture coords, so water doesnt look distorted when xWidth != zWidth
  glBegin(GL_QUADS);
  // The back left vertice for the water
  glMultiTexCoord2f(GL_TEXTURE0, 0, waterUV);                   // Reflection texture
  glMultiTexCoord2f(GL_TEXTURE1, 0, waterUV - move);            // Refraction texture
  glMultiTexCoord2f(GL_TEXTURE2, 0, normalUV + move2);          // Normal map texture
  glMultiTexCoord2f(GL_TEXTURE3, 0, 0);                         // DUDV map texture
  glVertex3f(this->waterVerts[0], this->waterHeight, this->waterVerts[1]);

  // The front left vertice for the water
  glMultiTexCoord2f(GL_TEXTURE0, 0, 0);                         // Reflection texture
  glMultiTexCoord2f(GL_TEXTURE1, 0, -move);                     // Refraction texture
  glMultiTexCoord2f(GL_TEXTURE2, 0, move2);                     // Normal map texture
  glMultiTexCoord2f(GL_TEXTURE3, 0, 0);                         // DUDV map texture
  glVertex3f(this->waterVerts[2], this->waterHeight, this->waterVerts[3]);

  // The front right vertice for the water
  glMultiTexCoord2f(GL_TEXTURE0, waterUV, 0);                   // Reflection texture
  glMultiTexCoord2f(GL_TEXTURE1, waterUV, -move);               // Refraction texture
  glMultiTexCoord2f(GL_TEXTURE2, normalUV, move2);              // Normal map texture
  glMultiTexCoord2f(GL_TEXTURE3, 0, 0);                         // DUDV map texture
  glVertex3f(this->waterVerts[4], this->waterHeight, this->waterVerts[5]);

  // The back right vertice for the water
  glMultiTexCoord2f(GL_TEXTURE0, waterUV, waterUV);             // Reflection texture
  glMultiTexCoord2f(GL_TEXTURE1, waterUV, waterUV - move);      // Refraction texture
  glMultiTexCoord2f(GL_TEXTURE2, normalUV, normalUV + move2);   // Normal map texture
  glMultiTexCoord2f(GL_TEXTURE3, 0, 0);                         // DUDV map texture
  glVertex3f(this->waterVerts[6], this->waterHeight, this->waterVerts[7]);
  glEnd();

  this->shader.deactivateShader();

  mat.unselect();

  glPopMatrix();
}

/**
 * @brief tick tack, calculates the flow of the water
 */
void MappedWater::tick(float dt)
{
  // makes the water flow
  this->move += this->waterFlow * dt;
  this->move2 = this->move * this->kNormalMapScale;

  // fading TODO fix this so it isnt hacky anymore
  if(bFadeWaterUV)
  {
    this->waterUVFader = new tAnimation<MappedWater>(this, &MappedWater::setWaterUV);
    this->waterUVFader->setInfinity(ANIM_INF_CONSTANT);

    this->waterUVFader->addKeyFrame(this->waterUV, this->waterUVFadeTime, ANIM_LINEAR);
    this->waterUVFader->addKeyFrame(this->newWaterUV, 0, ANIM_LINEAR);

    bFadeWaterUV = false;
    this->waterUVFader->replay();
  }

  if(bFadeWaterFlow)
  {
    this->waterFlowFader = new tAnimation<MappedWater>(this, &MappedWater::setWaterFlow);
    this->waterFlowFader->setInfinity(ANIM_INF_CONSTANT);

    this->waterFlowFader->addKeyFrame(this->waterFlow, this->waterFlowFadeTime, ANIM_LINEAR);
    this->waterFlowFader->addKeyFrame(this->newWaterFlow, 0, ANIM_LINEAR);

    bFadeWaterFlow = false;
    this->waterFlowFader->replay();
  }

  if(bFadeShineSize)
  {
    this->shineSizeFader = new tAnimation<MappedWater>(this, &MappedWater::resetShineSize);
    this->shineSizeFader->setInfinity(ANIM_INF_CONSTANT);

    this->shineSizeFader->addKeyFrame(this->shineSize, this->shineSizeFadeTime, ANIM_LINEAR);
    this->shineSizeFader->addKeyFrame(this->newShineSize, 0, ANIM_LINEAR);

    bFadeShineSize = false;
    this->shineSizeFader->replay();
  }

  if(bFadeShineStrength)
  {
    this->shineStrengthFader = new tAnimation<MappedWater>(this, &MappedWater::resetShineStrength);
    this->shineStrengthFader->setInfinity(ANIM_INF_CONSTANT);

    this->shineStrengthFader->addKeyFrame(this->shineStrength, this->shineStrengthFadeTime, ANIM_LINEAR);
    this->shineStrengthFader->addKeyFrame(this->newShineStrength, 0, ANIM_LINEAR);

    bFadeShineStrength = false;
    this->shineStrengthFader->replay();
  }

  if(bFadeReflStrength)
  {
    this->reflStrengthFader = new tAnimation<MappedWater>(this, &MappedWater::resetReflStrength);
    this->reflStrengthFader->setInfinity(ANIM_INF_CONSTANT);

    this->reflStrengthFader->addKeyFrame(this->reflStrength, this->reflStrengthFadeTime, ANIM_LINEAR);
    this->reflStrengthFader->addKeyFrame(this->newReflStrength, 0, ANIM_LINEAR);

    bFadeReflStrength = false;
    this->reflStrengthFader->replay();
  }

  if(bFadeRefraction)
  {
    this->refractionFader = new tAnimation<MappedWater>(this, &MappedWater::resetRefraction);
    this->refractionFader->setInfinity(ANIM_INF_CONSTANT);

    this->refractionFader->addKeyFrame(this->refraction, this->refractionFadeTime, ANIM_LINEAR);
    this->refractionFader->addKeyFrame(this->newRefraction, 0, ANIM_LINEAR);

    bFadeRefraction = false;
    this->refractionFader->replay();
  }

  if(bFadeWaterHeight)
  {
    this->waterHeightFader = new tAnimation<MappedWater>(this, &MappedWater::setWaterHeight);
    this->waterHeightFader->setInfinity(ANIM_INF_CONSTANT);

    this->waterHeightFader->addKeyFrame(this->waterHeight, this->waterHeightFadeTime, ANIM_LINEAR);
    this->waterHeightFader->addKeyFrame(this->newWaterHeight, 0, ANIM_LINEAR);

    bFadeWaterHeight = false;
    this->waterHeightFader->replay();
  }

  if(bFadeWaterColor)
  {
    this->waterColorRFader = new tAnimation<MappedWater>(this, &MappedWater::resetWaterColorR);
    this->waterColorRFader->setInfinity(ANIM_INF_CONSTANT);

    this->waterColorRFader->addKeyFrame(this->waterColor.x, this->waterColorFadeTime, ANIM_LINEAR);
    this->waterColorRFader->addKeyFrame(this->newWaterColor.x, 0, ANIM_LINEAR);

    this->waterColorRFader->replay();

    this->waterColorGFader = new tAnimation<MappedWater>(this, &MappedWater::resetWaterColorG);
    this->waterColorGFader->setInfinity(ANIM_INF_CONSTANT);

    this->waterColorGFader->addKeyFrame(this->waterColor.y, this->waterColorFadeTime, ANIM_LINEAR);
    this->waterColorGFader->addKeyFrame(this->newWaterColor.y, 0, ANIM_LINEAR);

    this->waterColorGFader->replay();

    this->waterColorBFader = new tAnimation<MappedWater>(this, &MappedWater::resetWaterColorB);
    this->waterColorBFader->setInfinity(ANIM_INF_CONSTANT);

    this->waterColorBFader->addKeyFrame(this->waterColor.z, this->waterColorFadeTime, ANIM_LINEAR);
    this->waterColorBFader->addKeyFrame(this->newWaterColor.z, 0, ANIM_LINEAR);

    this->waterColorBFader->replay();

    bFadeWaterColor = false;
  }
}

/**
 * @brief prepares everything to render the reflection texutre
 */
void MappedWater::activateReflection()
{
  // To create the reflection texture we just need to set the view port
  // to our texture map size, then render the current scene our camera
  // is looking at to the already allocated texture unit.  Since this
  // is a reflection of the top of the water surface we use clipping
  // planes to only render the top of the world as a reflection.  If
  // we are below the water we don't flip the reflection but just use
  // the current view of the top as we are seeing through the water.
  // When you look through water at the surface it isn't really reflected,
  // only when looking down from above the water on the surface.

  // save viewport matrix and change the viewport size
  glPushAttrib(GL_VIEWPORT_BIT);
  glViewport(0,0, textureSize, textureSize);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  // If our camera is above the water we will render the scene flipped upside down.
  // In order to line up the reflection nicely with the world we have to translate
  // the world to the position of our reflected surface, multiplied by two.
  glEnable(GL_CLIP_PLANE0);
  Vector pos = State::getCameraNode()->getAbsCoor();

  if(pos.y > waterHeight)
  {
    // Translate the world, then flip it upside down
    glTranslatef(0, waterHeight * 2, 0);
    glScalef(1, -1, 1);

    // Since the world is updside down we need to change the culling to FRONT
    glCullFace(GL_FRONT);

    // Set our plane equation and turn clipping on
    double plane[4] = {0, 1, 0, -waterHeight};
    glClipPlane(GL_CLIP_PLANE0, plane);
  }
  else
  {
    // If the camera is below the water we don't want to flip the world,
    // but just render it clipped so only the top is drawn.
    double plane[4] = {0, 1, 0, waterHeight};
    glClipPlane(GL_CLIP_PLANE0, plane);
  }
}

/**
 * @brief ends the reflection and saves the graphic buffer into a texture
 */
void MappedWater::deactivateReflection()
{
  glDisable(GL_CLIP_PLANE0);
  glCullFace(GL_BACK);

  // Create the texture and store it on the video card
  mat.renderToTexture(0, GL_TEXTURE_2D, 0, 0, 0, 0, 0, textureSize, textureSize);

  glPopMatrix();
  glPopAttrib();
}

/**
 * @brief prepares everything to render the refraction texutre
 */
void MappedWater::activateRefraction()
{
  // To create the refraction and depth textures we do the same thing
  // we did for the reflection texture, except we don't need to turn
  // the world upside down.  We want to find the depth of the water,
  // not the depth of the sky and above water terrain.

  // save viewport matrix and change the viewport size
  glPushAttrib(GL_VIEWPORT_BIT);
  glViewport(0,0, textureSize, textureSize);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  // If our camera is above the water we will render only the parts that
  // are under the water.  If the camera is below the water we render
  // only the stuff above the water.  Like the reflection texture, we
  // incorporate clipping planes to only render what we need.

  // If the camera is above water, render the data below the water
  glEnable(GL_CLIP_PLANE0);
  Vector pos = State::getCameraNode()->getAbsCoor();
  if(pos.y > waterHeight)
  {
    double plane[4] = {0, -1, 0, waterHeight};
    glClipPlane(GL_CLIP_PLANE0, plane);
  }
  // If the camera is below the water, only render the data above the water
  else
  {
    glCullFace(GL_FRONT);
    double plane[4] = {0, 1, 0, -waterHeight};
    glClipPlane(GL_CLIP_PLANE0, plane);
  }
}

/**
 * @brief ends the refraction and saves the graphic buffer into a texture
 */
void MappedWater::deactivateRefraction()
{
  glDisable(GL_CLIP_PLANE0);
  glCullFace(GL_BACK);

  // Create the texture and store it on the video card
  mat.renderToTexture(1, GL_TEXTURE_2D, 0, 0, 0, 0, 0, textureSize, textureSize);

  glPopMatrix();
  glPopAttrib();
}
