/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "material.h"

#include "texture.h"
#include "debug.h"
#include "compiler.h"

#include "loading/load_param.h"

#include "resource_texture.h"

ObjectListDefinition(Material);

/**
 * @brief creates a Material.
 * @param mtlName Name of the Material to be added to the Material List
 */
Material::Material (const std::string& mtlName)
{
  this->registerObject(this, Material::_objectList);

  this->setIllum(3);
  this->setDiffuse(1,1,1);
  this->setAmbient(0,0,0);
  this->setSpecular(.5,.5,.5);
  this->setShininess(2.0);
  this->setTransparency(1.0);

  this->ambientTexture = NULL;
  this->specularTexture = NULL;
  this->sFactor = GL_SRC_ALPHA;
  this->tFactor = GL_ONE;

  this->setName(mtlName);
}

Material& Material::operator=(const Material& material)
{
  this->illumModel = material.illumModel;
  this->diffuse = material.diffuse;
  this->specular = material.specular;
  this->ambient = material.ambient;
  this->shininess = material.shininess;

  this->textures = material.textures;
  this->sFactor = material.sFactor;
  this->tFactor = material.tFactor;
  this->setName(material.getName());

  return *this;
}



void Material::loadParams(const TiXmlElement* root)
{
  LoadParam(root, "illum", this, Material, setIllum);

  LoadParam(root, "diffuse-color", this, Material , setDiffuse);
  LoadParam(root, "ambient-color", this, Material , setAmbient);
  LoadParam(root, "specular-color", this, Material , setSpecular);
  LoadParam(root, "transparency", this, Material , setTransparency);

  LoadParam(root, "tex", this, Material, setDiffuseMap);
  LoadParam(root, "blendfunc", this, Material, setBlendFuncS)
  .defaultValues("ZERO", "ZERO");
}


/**
 * @brief deletes a Material
 */
Material::~Material()
{
  PRINTF(5)("delete Material %s.\n", this->getCName());

  if (this == Material::selectedMaterial)
    Material::selectedMaterial = NULL;
}


const Material* Material::selectedMaterial = NULL;

/**
 * @brief sets the material with which the following Faces will be painted
 */
bool Material::select() const
{
  if (unlikely(this == Material::selectedMaterial))
    return true;

  /// !!  HACK !!! FIX THIS IN MODEL ///
  else if (likely(Material::selectedMaterial != NULL))
  {
    Material::unselect();
    //     for(unsigned int i = 0; i < Material::selectedMaterial->textures.size(); ++i)
    //     {
    //         glActiveTexture(Material::glTextureArbs[i]);
    //         glBindTexture(GL_TEXTURE_2D, 0);
    //         glDisable(GL_TEXTURE_2D);
    //     }
  }

  if (likely(Material::selectedMaterial != NULL))
  {
    for(unsigned int i = 0; i < Material::selectedMaterial->textures.size(); ++i)
    {
      if (GLEW_ARB_multitexture)
        glActiveTexture(Material::glTextureArbs[i]);
      //glBindTexture(GL_TEXTURE_2D, 0);
      glDisable(GL_TEXTURE_2D);
    }
  }

  // setting diffuse color
  glColor4f (diffuse.r(), diffuse.g(), diffuse.b(), diffuse.a());
  // setting ambient color
  glMaterialfv(GL_FRONT, GL_AMBIENT, &this->ambient[0]);
  // setting up Sprecular
  glMaterialfv(GL_FRONT, GL_SPECULAR, &this->specular[0]);
  // setting up Shininess
  glMaterialf(GL_FRONT, GL_SHININESS, this->shininess);

  // setting the transparency
  if (this->diffuse.a() < 1.0 ||       /* This allows alpha blending of 2D textures with the scene */
      (likely(!this->textures.empty() && this->textures[0].hasAlpha())))
  {
    glEnable(GL_BLEND);
    glBlendFunc(this->sFactor, this->tFactor);
  }
  else
  {
    glDisable(GL_BLEND);
  }


  // setting illumination Model
  if (this->illumModel == 1) //! @todo make this work, if no vertex-normals are read.
    glShadeModel(GL_FLAT);
  else if (this->illumModel >= 2)
    glShadeModel(GL_SMOOTH);

  for(unsigned int i = 0; i < this->textures.size(); ++i)
  {
    if (GLEW_ARB_multitexture)
      glActiveTexture(Material::glTextureArbs[i]);
    glEnable(GL_TEXTURE_2D);
    if(this->textures[i].hasAlpha())
    {
      glEnable(GL_BLEND);
    }
    glBindTexture(GL_TEXTURE_2D, this->textures[i].getTexture());
  }
  Material::selectedMaterial = this;

  return true;
}
/**
 * @brief Deselect Material (if one is selected).
 */
void Material::unselect()
{
  Material::selectedMaterial = NULL;
  for(unsigned int i = 0; i < 8; ++i)
  {
    glActiveTexture(Material::glTextureArbs[i]);
    glBindTexture(GL_TEXTURE_2D, 0);
    glDisable(GL_TEXTURE_2D);
  }
}

/**
 * @brief Sets the Material Illumination Model.
 * @param illu illumination Model in int form
 */
void Material::setIllum (int illum)
{
  PRINTF(4)("setting illumModel of Material %s to %i\n", this->getCName(), illum);
  this->illumModel = illum;
}

/**
 * @brief Sets the Material Diffuse Color.
 * @param r Red Color Channel.a
 * @param g Green Color Channel.
 * @param b Blue Color Channel.
 */
void Material::setDiffuse (float r, float g, float b)
{
  PRINTF(4)("setting Diffuse Color of Material %s to r=%f g=%f b=%f.\n", this->getCName(), r, g, b);
  this->diffuse = Color(r, g, b, this->diffuse.a() );
}


/**
 * @brief Sets the Material Ambient Color.
 * @param r Red Color Channel.
 * @param g Green Color Channel.
 * @param b Blue Color Channel.
*/
void Material::setAmbient (float r, float g, float b)
{
  PRINTF(4)("setting Ambient Color of Material %s to r=%f g=%f b=%f.\n", this->getCName(), r, g, b);
  this->ambient = Color(r, g, b, 1.0);
}

/**
 * @brief Sets the Material Specular Color.
 * @param r Red Color Channel.
 * @param g Green Color Channel.
 * @param b Blue Color Channel.
 */
void Material::setSpecular (float r, float g, float b)
{
  PRINTF(4)("setting Specular Color of Material %s to r=%f g=%f b=%f.\n", this->getCName(), r, g, b);
  this->specular = Color (r, g, b, 1.0);
}

/**
 * @brief Sets the Material Shininess.
 * @param shini stes the Shininess from float.
*/
void Material::setShininess (float shini)
{
  this->shininess = shini;
}

/**
 * @brief Sets the Material Transparency.
 * @param trans stes the Transparency from int.
*/
void Material::setTransparency (float trans)
{
  PRINTF(4)("setting Transparency of Material %s to %f.\n", this->getCName(), trans);
  this->diffuse.a() = trans;
}

/**
 * @brief sets the Blend-Function Parameters
 * @param sFactor the Source Parameter.
 * @param tFactor the Desitnation Parameter.
 */
void Material::setBlendFuncS(const std::string& sFactor, const std::string& tFactor)
{
  this->setBlendFunc(Material::stringToBlendFunc(sFactor), Material::stringToBlendFunc(tFactor));
}


/**
 * @brief Sets the Diffuse map of this Texture.
 * @param texture The Texture to load
 * @param textureNumber The Texture-Number from 0 to GL_MAX_TEXTURE_UNITS
 */
void Material::setDiffuseMap(const Texture& texture, unsigned int textureNumber)
{
  assert(textureNumber < Material::getMaxTextureUnits());

  if (this->textures.size() <= textureNumber)
    this->textures.resize(textureNumber+1, Texture());

  //! @todo check if RESOURCE MANAGER is availiable
  this->textures[textureNumber] = texture;
}

/**
 * @brief Sets the Diffuse map of this Texture by a Texture-pointer.
 * @param textureDataPointer The Texture-Data-Pointer to load.
 * @param textureNumber The Texture-Number from 0 to GL_MAX_TEXTURE_UNITS
 */
void Material::setDiffuseMap(const TextureData::Pointer& textureDataPointer, unsigned int textureNumber)
{
  assert(textureNumber < Material::getMaxTextureUnits());

  if (this->textures.size() <= textureNumber)
    this->textures.resize(textureNumber+1, Texture());

  this->textures[textureNumber] = textureDataPointer;
}


/**
 * @brief Sets the Materials Diffuse Map
 * @param dMap the Name of the Image to Use
 * @param textureNumber The Texture-Number from 0 to GL_MAX_TEXTURE_UNITS
 */
void Material::setDiffuseMap(const std::string& dMap, GLenum target, unsigned int textureNumber)
{
  assert(textureNumber < Material::getMaxTextureUnits());

  PRINTF(5)("setting Diffuse Map %s\n", dMap.c_str());
  if (this->textures.size() <= textureNumber)
    this->textures.resize(textureNumber+1, Texture());

  //! @todo check if RESOURCE MANAGER is availiable
  if (!dMap.empty())
  {
    this->textures[textureNumber] = ResourceTexture(dMap);
    //dynamic_cast<Texture*>(ResourceManager::getInstance()->load(dMap, IMAGE, RP_GAME, (int)target));
    /*    if (tex != NULL)
          this->textures[textureNumber] = tex;
        else
          this->textures[textureNumber] = Texture();*/
  }
  else
  {
    this->textures[textureNumber] = Texture();
  }
}

/**
 * @brief Sets the Materials Diffuse Map
 * @param surface pointer to SDL_Surface which shall be used as texture.
 * @param target the GL-Target to load the Surface to.
 * @param textureNumber The Texture-Number from 0 to GL_MAX_TEXTURE_UNITS.
 */
void Material::setSDLDiffuseMap(SDL_Surface *surface, GLenum target, unsigned int textureNumber)
{
  assert(textureNumber < Material::getMaxTextureUnits());


  if (this->textures.size() <= textureNumber)
    this->textures.resize(textureNumber+1, Texture());

  if(surface != NULL)
  {
    this->textures[textureNumber] = Texture(surface, GL_TEXTURE_2D);
  }
  else
  {
    this->textures[textureNumber] = Texture();
  }

}

/**
 * @brief Renders to a Texture.
 * @param textureNumber The Texture-Number from 0 to GL_MAX_TEXTURE_UNITS.
 * @param target The GL-Target.
 * @param level the MipMap-Level to render to.
 * @param xoffset The offset in the Source from the left.
 * @param yoffset The offset in the Source from the top (or bottom).
 * @param x The Offset in the Destination from the left.
 * @param y The Offset in the Destination from the top (or bottom).
 * @param width The width of the region to copy.
 * @param height The height of the region to copy.
 */
void Material::renderToTexture(unsigned int textureNumber, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height)
{
  assert(textureNumber < Material::getMaxTextureUnits());
  assert(textureNumber < this->textures.size());

  // HACK
  glActiveTexture(textureNumber);
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, this->textures[textureNumber].getTexture());
  glCopyTexSubImage2D(target, level, xoffset, yoffset, x, y, width, height);

}

/**
 * @brief Sets the Materials Ambient Map
 * @todo implement this
 */
void Material::setAmbientMap(const std::string& aMap, GLenum target)
{
  /// FIXME SDL_Surface* ambientMap;

}

/**
 * @brief Sets the Materials Specular Map
 * @param sMap the Name of the Image to Use
 * @todo implement this
 */
void Material::setSpecularMap(const std::string& sMap, GLenum target)
{
  /// FIXME SDL_Surface* specularMap;

}

/**
 * @brief Sets the Materials Bumpiness
 * @param bump the Name of the Image to Use
 * @todo implemet this
 */
void Material::setBump(const std::string& bump)
{}


/**
 * @returns the Maximim Texture Unit the users OpenGL-Implementation supports.
 */
unsigned int Material::getMaxTextureUnits()
{
  int maxTexUnits = 0;
  glGetIntegerv(GL_MAX_TEXTURE_UNITS, &maxTexUnits);
  return maxTexUnits;
}



void Material::debug() const
{
  PRINT(0)("Debug Material: %s\n", this->getCName());
  PRINT(0)("illumModel: %d ; ShiniNess %f\n", this->illumModel, shininess);
  PRINT(0)("diffuseColor: "); diffuse.debug();
  PRINT(0)("ambientColor: "); ambient.debug();
  PRINT(0)("diffuseColor: "); specular.debug();
  PRINT(0)("Blending Properties: Source: %s, Destination: %s\n", blendFuncToString(sFactor).c_str(), blendFuncToString(tFactor).c_str());

  PRINT(0)("Textures: %d loaded", textures.size());
  if (!this->textures.empty())
  {
    PRINT(0)(" - ID's: ");
    for (unsigned int i = 0; i < textures.size(); ++i)
    {
      PRINT(0)("%d ", textures[i].getTexture());
    }
  }
  PRINT(0)("\n");
}


const GLenum Material::glTextureArbs[] =
  {
    GL_TEXTURE0,
    GL_TEXTURE1,
    GL_TEXTURE2,
    GL_TEXTURE3,
    GL_TEXTURE4,
    GL_TEXTURE5,
    GL_TEXTURE6,
    GL_TEXTURE7
  };


/**
 * @param blendFunc the Enumerator to convert to a String.
 * @returns the matching String if found
 */
const std::string& Material::blendFuncToString(GLenum blendFunc)
{
  for (unsigned int i = 0; i < 9; ++i)
    if (blendFunc == Material::glBlendFuncParams[i])
      return Material::blendFuncNames[i];
  PRINTF(2)("Supplied an Invalid Enumerator to blendfunc %d\n", blendFunc);
  return Material::blendFuncNames[0];
}

/**
 * @param blendFuncString the String to convert into a gl-enumeration
 * @returns the matching GL-enumeration if found.
 */
GLenum Material::stringToBlendFunc(const std::string& blendFuncString)
{
  for (unsigned int i = 0; i < 9; ++i)
    if (blendFuncString == Material::blendFuncNames[i])
      return Material::glBlendFuncParams[i];
  PRINTF(2)("BlendFunction %s not recognized using %s\n", blendFuncString.c_str(), Material::blendFuncNames[0].c_str());
  return Material::glBlendFuncParams[0];
}


const GLenum Material::glBlendFuncParams[] =
  {
    GL_ZERO,
    GL_ONE,
    GL_DST_COLOR,
    GL_ONE_MINUS_DST_COLOR,
    GL_SRC_ALPHA,
    GL_ONE_MINUS_SRC_ALPHA,
    GL_DST_ALPHA,
    GL_ONE_MINUS_DST_ALPHA,
    GL_SRC_ALPHA_SATURATE
  };

const std::string Material::blendFuncNames[] =
  {
    "ZERO",
    "ONE",
    "DST_COLOR",
    "ONE_MINUS_DST_COLOR",
    "SRC_ALPHA",
    "ONE_MINUS_SRC_ALPHA",
    "DST_ALPHA",
    "ONE_MINUS_DST_ALPHA",
    "SRC_ALPHA_SATURATE"

  };
