/*!
 * @file ini_parser.h
 * A small ini file parser
 *
 * Can be used to find a defined [Section] in an ini file and get the VarName = Value entries
 */

#ifndef _INI_PARSER_H
#define _INI_PARSER_H

#define PARSELINELENGHT     512       //!< how many chars to read at once

#include <string>
#include <list>

//! ini-file parser
/**
 * This class can be used to load an initializer file and parse it's contents for variablename=value pairs.
 */
class IniParser
{
public:
  ////////////////////////////////////
  /// A Node for a Ini-Node. The base of all INI-elements.
  class Node
  {
  public:
    Node(const std::string& name, const std::string& comment = "");
    virtual ~Node() {};
    const std::string& name() const  { return _name; };
    const std::string& comment() const  { return _comment; };
    void setName(const std::string& name) { this->_name = name; };
    void setComment(const std::string& comment) { this->_comment = comment; };

    bool operator==(const std::string& name) const { return _name == name; };
    bool operator==(const Node& node) const { return this->_name == node._name; };

    virtual void debug() const = 0;

  private:
    std::string         _comment;  //!< A Comment that is appendet to the Top of this Node
    std::string         _name;     //!< name of a given Node
  };

  //! a class for Entries in the Parser's File's Sections
class Entry : public Node
  {
  public:
    Entry(const std::string& name, const std::string& value = "", const std::string& comment = "");
    const std::string& value() const { return _value; };
    void setValue (const std::string& value) { _value = value; };

    virtual void debug() const;

  public:
    typedef std::list<Entry>                    list;
    typedef list::iterator                      iterator;
    typedef list::const_iterator                const_iterator;

  private:
    std::string         _value;    //!< value of a given Entry
  };

  //! a clas for Sections in the Parser's file
class Section : public Node
  {
  public:
    Section(const std::string& sectionName, const std::string& comment = "");

    Entry& addEntry(const std::string& entryName, const std::string& value = "", const std::string& comment = "");
    bool editEntry(const std::string& entryName, const std::string& value, bool createMissing = true);
    const std::string& getValue(const std::string& entryName, const std::string& defaultValue = "") const;

    bool setEntryComment(const std::string& entryName, const std::string& comment);
    const std::string& getEntryComment(const std::string& entryName) const;

    const Entry::list& entries() const { return _entries; }
    Entry* getEntry(const std::string& entryName);

    Entry::const_iterator getEntryIt(const std::string& entryName) const;
    Entry::iterator begin() { return _entries.begin(); };
    Entry::const_iterator begin() const { return _entries.begin(); };
    Entry::iterator end() { return _entries.end(); };
    Entry::const_iterator end() const { return _entries.end(); };

    void clear();

    virtual void debug() const;

  public:
    typedef std::list<Section>                  list;
    typedef list::iterator                      iterator;
    typedef list::const_iterator                const_iterator;

  private:
    Entry::list     _entries;  //!< a list of entries for this section
  };

  //! A class for a INI-file.
class Document : public Node
  {
  public:
    Document(const std::string& fileName, const std::string& comment = "");

    Section& addSection(const std::string& sectionName, const std::string& comment = "");
    bool removeSection(const std::string& sectionName);
    bool setSectionComment(const std::string& sectionName, const std::string& comment);

    const Section::list& sections() const { return _sections; }
    Section* getSection(const std::string& sectionName);

    Section::const_iterator getSectionIt(const std::string& sectionName) const;
    Section::iterator begin() { return _sections.begin(); };
    Section::const_iterator begin() const { return _sections.begin(); };
    Section::iterator end() { return _sections.end(); };
    Section::const_iterator end() const { return _sections.end(); };

    bool addEntry(const std::string& sectionName, const std::string& entryName, const std::string& value = "", const std::string& comment = "");
    bool editEntry(const std::string& sectionName, const std::string& entryName, const std::string& value, bool createMissing = true);
    const std::string& getValue(const std::string& sectionName, const std::string& entryName, const std::string& defaultValue = "") const;

    bool setEntryComment(const std::string& sectionName, const std::string& entryName, const std::string& comment);
    const std::string& getEntryComment(const std::string& sectionName, const std::string& entryName) const;

    void clear();

    virtual void debug() const;

  private:
    Section::list            _sections;        //!< a list of all stored Sections of the Parser.
  };

public:
  IniParser (const std::string& filename = "");
  virtual ~IniParser ();

  /** @returns true if the file is opened, false otherwise */
  bool isOpen() const { return !this->_document.sections().empty(); };
  /** @returns the fileName we have opened. */
  const std::string& getFileName() const { return this->_document.name(); };

  /// Read and Write the File
  bool readFile(const std::string& fileName, bool keepSettings = false);
  bool writeFile(const std::string& fileName) const;

  void setFileComment(const std::string& fileComment);
  const std::string& getFileComment() const { return this->_document.comment(); };

  /// Woring with Sections.
  Section& addSection(const std::string& sectionName);
  // iterate through sections with these Functions
  Section* getSection(const std::string& sectionName) { return this->_document.getSection(sectionName); };
  Section::const_iterator getSectionIt(const std::string& sectionName) const;

  Section::iterator begin() { return this->_document.begin(); };
  Section::const_iterator begin() const { return this->_document.begin(); };
  Section::iterator end() { return this->_document.end(); };
  Section::const_iterator end() const { return this->_document.end(); };

  void setSectionComment(const std::string& sectionName, const std::string& comment);
  const std::string& getSectionsComment(const std::string& sectionNane) const;

  /// Working on Entries. (globally)
  bool addEntry(const std::string& sectionName, const std::string& entryName, const std::string& value, const std::string& comment);
  const std::string& getValue(const std::string& sectionNmae, const std::string& entryName, const std::string& defaultValue = "") const;
  bool editEntry(const std::string& sectionName, const std::string& entryName, const std::string& value, bool createMissing = true);

  void setEntryComment(const std::string& sectionName, const std::string& entryName, const std::string& comment);
  const std::string& getEntryComment(const std::string& sectionName, const std::string& entryName) const;

  // maintenance.
  void debug() const;

private:
  void setNodeComment(Node* node, std::list<std::string>* comments);
private:
  std::string                      _fileName;
  Document                         _document;

  static const std::string         _emptyString;     //!< Just an Empty String that will be returned if nothing else is found.
};

#endif /* _INI_PARSER_H */
