/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

   Borrowed the Main code from 'Bart Vanhauwaert' (license below)
   and adopted it to my likings.
*/

/**
 * Copyright (C) 2002 Bart Vanhauwaert
 *
 * Permission to use, copy, modify, distribute and sell this software
 * for any purpose is hereby granted without fee. This license
 * includes (but is not limited to) standalone compilation or as part
 * of a larger project.
 *
 * This software is provided "as is" without express or implied warranty.
 *
 * For a full statement on warranty and terms and conditions for
 * copying, distribution and modification, please see the comment block
 * at the end of this file.
 *
 * Version 1
 *
 */

#include "directory.h"

#if not defined (__WIN32__)
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
const char Directory::delimiter = '/';
#else
#include <windows.h>
#include <winbase.h>
const char Directory::delimiter = '\\';
#endif

#include <iostream>

/**
 * @brief Constructs a Directory handler.
 * @param directoryName the name of the Directory to access.
 */
Directory::Directory(const std::string& directoryName)
    : File(directoryName)
{
  this->_opened = false;
}

/**
 * @brief construct a Copy of directory.
 * @param directory the Directory to copy.
 */
Directory::Directory(const Directory& directory)
    : File(directory)
{
  this->_opened = directory._opened;
  this->_fileNames = directory._fileNames;
}


/**
 * @brief destructs the Directory.
 */
Directory::~Directory()
{
  this->close();
}


/**
 * @brief openes the Directory
 * @returns true on success, false on error. (does not exist, no rights -> test with functions of File)
 *
 * Fills the List of Files, and sets the Directory to open state
 */
bool Directory::open()
{
  if (this->_opened)
    this->close();

  // Openes the Directory for reading:
#if not defined(__WIN32__)
  DIR* handle;
  handle = opendir(this->name().c_str());
  if (!handle)
  {
    std::cerr << "could not open directory " << this->name() << " for reading" << std::endl;
    return false;
  }

  dirent* entry;
  while ((entry = readdir(handle)) != NULL)
    this->_fileNames.push_back(entry->d_name);
  closedir(handle);

#else
  HANDLE handle;

  // First check the attributes trying to access a non-Directory with
  // FindFirstFile takes ages
  DWORD attrs = GetFileAttributes(this->name().c_str());
  if ( (attrs == 0xFFFFFFFF) || ((attrs && FILE_ATTRIBUTE_DIRECTORY) == 0) )
  {
    return false;
  }
  std::string Full(this->name());
  // Circumvent a problem in FindFirstFile with c:\\* as parameter
  if ( (Full.length() > 0) && (Full[Full.length()-1] != '\\') )
    Full += "\\";
  WIN32_FIND_DATA entry;
  handle = FindFirstFile( (Full+"*").c_str(), &entry);
  if (handle == INVALID_HANDLE_VALUE)
  {
    std::cerr << "could not open directory " << this->name() << " for reading" << std::endl;
    return false;
  }
  else
  {
    this->_fileNames.push_back(entry.cFileName);
  }
  int ok;
  while ((ok = FindNextFile(handle, &entry)) != 0)
    this->_fileNames.push_back(entry.cFileName);
  FindClose(handle);
#endif /* __WIN_32__ */

  this->_opened = true;
  return true;
}

/**
 * @brief closes the directory
 * @returns true.
 *
 * Clears the List of Files in the Directory.
 */
bool Directory::close()
{
  this->_opened = false;
  this->_fileNames.clear();
  return true;
}


/**
 * @brief creates the directory
 * @returns true on success, false on error
 */
bool Directory::create()
{
#if not defined (__WIN32__)
  return (!mkdir(this->name().c_str(), 0777));
#else
  return (!CreateDirectory(this->name().c_str(), NULL));
#endif
}


Directory Directory::operator+(const Directory& dir) const
{
  return Directory(*this) += dir;
}

/**
 * @param dir the Directory to append to this one (say this one is "/var", then dir can be "log")
 * @returns The Directory appended by dir.
 *
 * @note the Directoy will again be closed even if it was opened previously!
 */
Directory& Directory::operator+=(const Directory& dir)
{
  this->setFileName(this->name() + Directory::delimiter + dir.name());
  return *this;
}

/**
 * @brief Traverses the Directory tree one step up. (Parent Directory)
 * @returns a Reference to the Directory.
 */
Directory& Directory::operator--()
{
}


/**
 * @brief Traverses the Directory tree one step up. (Parent Directory)
 * @param int the PostFix iterator
 * @returns a Reference to the Directory.
 */
Directory& Directory::operator--(int)
{
}

/**
 * @returns The Parent Directory.
 */
Directory Directory::parentDir() const
{

}

File operator+(const Directory& dir, const File& file)
{
  return File(dir.name() + Directory::delimiter + file.name());
}
