/*!
 * @file event_handler.h
 * Definition of the EventHandler
 *
 */

#ifndef _EVENT_HANDLER_H
#define _EVENT_HANDLER_H

#include "base_object.h"
#include "key_mapper.h"
#include "event_def.h"
#include "event.h"
#include <stack>
#include <vector>

// FORWARD DECLARATION
class EventListener;

//! The one Event Handler from Orxonox
class EventHandler : public BaseObject
{
  ObjectListDeclaration(EventHandler);
public:
  virtual ~EventHandler();
  /** @returns a Pointer to the only object of this Class */
  inline static EventHandler* getInstance() { if (!singletonRef) singletonRef = new EventHandler();  return singletonRef; };
  void init();

  void setState(elState state);
  /** @returns the current state */
inline elState getState() const { return this->state; };

  void pushState(elState state);
  elState popState();

  void subscribe(EventListener* el, elState state, int eventType);
  void unsubscribe(EventListener* el, elState state, int eventType);
  void unsubscribe(EventListener* el, elState state = ES_ALL);
  void flush(elState state);
  /** @returns true, if the @param state has @param eventType subscribed?? */
  bool isSubscribed(elState state, int eventType);

  /** @param key the key to check @returns true if key is pressed */
  bool isPressed(SDLKey key) const { return this->keyState[key]; };

  void withUNICODE(elState state, bool enableUNICODE);
  void grabEvents(bool grabEvents);
  bool grabbedEvents() const { return this->eventsGrabbed; };

  void process() const;
  void dispachEvent(elState state, const Event& event) const;

  static int eventFilter(const SDL_Event *event);
  void debug() const;

  static const std::string& ELStateToString(elState state);
  static elState StringToELState(const std::string& stateName);

  static int releaseMouse(void* p);

private:
  EventHandler();

  bool findListener(std::vector<EventListener*>::iterator* it, elState state, int eventType, EventListener* listener);

public:
  static const std::string     stateNames[];

private:
  static EventHandler*         singletonRef;                    //!< the singleton reference

  std::vector<EventListener*>  listeners[ESTATE_NUMBER][EV_NUMBER]; //!< a list of registered listeners.
  elState                      state;                           //!< the state of the event handlder.
  std::stack<short>            stateStack;                      //!< a stack for the States we are in.
  KeyMapper                    keyMapper;                       //!< reference to the key mapper.

  bool                         bUNICODE[ESTATE_NUMBER];             //!< If unicode should be enabled.
  bool                         eventsGrabbed;                   //!< If the events should be grabbed

  Uint8*                       keyState;
};


#endif /* _EVENT_HANDLER_H */
