/*!
 * @file resource.h
 * @brief Definition of a Resource.
*/

#ifndef _RESOURCE_H
#define _RESOURCE_H

#include "base_object.h"
#include <string>
#include <vector>
#include <set>

#include "filesys/directory.h"

//! A Namespace Resources and ResourceHandling is defined in.
namespace Resources
{
  //! The KeepLevel handles the unloading of Resources.
  /**
   * Allocating a Resource also appends a KeepLevel to the Resource.
   * When the Resource is not used anymore it is decided on the grounds of the KeepLevel,
   * if the Resource should be deleted. (e.g. at the end of a Level, Campaign, or something like this).
   */
  class KeepLevel
  {
  public:
    KeepLevel();
    KeepLevel(unsigned int keepLevel);
    KeepLevel(const std::string& keepLevelName);

    //! Compare equality
    inline bool operator==(const KeepLevel& keepLevel) const { return this->_keepLevel == keepLevel._keepLevel; };
    //! Compares inequality
    inline bool operator!=(const KeepLevel& keepLevel) const { return this->_keepLevel != keepLevel._keepLevel; };
    //! Compares less/equal than
    inline bool operator<=(const KeepLevel& keepLevel) const { return this->_keepLevel <= keepLevel._keepLevel; };
    //! Compares less than
    inline bool operator<(const KeepLevel& keepLevel) const { return this->_keepLevel < keepLevel._keepLevel; };

    /** @returns the KeepLevel as a number */
    inline unsigned int keepLevel() const { return _keepLevel; };
    const std::string& name() const;
  private:
    unsigned int                _keepLevel;              //!< The KeepLevel a Resource is in.
  };


  ///////////////////
  // STORE POINTER //
  ///////////////////
  //! Stores a Resource-Pointer, the LoadString and it's keepLevel.
  class StorePointer
  {
  public:
    //! Virtual Destructor, that removes the Stored information-pointer.
    virtual ~StorePointer();

    /** @returns the LoadString this resource was loaded with */
    const std::string& loadString() const { return _loadString; };
    /** @returns the KeepLevel of this resource */
    const Resources::KeepLevel& keepLevel() const { return _keepLevel; };

    virtual bool last() const = 0;

    protected:
      StorePointer(const std::string& loadString, const Resources::KeepLevel& keeplevel);

    private:
      StorePointer(const StorePointer&) : _keepLevel(0) {};

  private:
    std::string                 _loadString;             //!< An identifier, to match when loading a File.
    Resources::KeepLevel        _keepLevel;              //!< The Priority of this resource. (can only be increased, so none else will delete this)
  };



  ///////////////////
  // RESOURCE TYPE //
  ///////////////////
  //! A Type of Resources.
  /**
   * The Type is used to store the Pointers to already loaded Resources,
   * and also to store type-specific properties.
   * These are the Loading Paths, the subpaths and so on.
   */
  class Type
  {
  public:
    virtual ~Type();
    /** @returns true if the names match @param typeName the Name to compare. @brief compare the Type with a Name */
    bool operator==(const std::string& typeName) const { return this->_typeName == typeName; };

    ////////////////////
    //// EXTENSIONS ////
    void addExtension(const std::string& extension);

    ///////////////
    //// PATHS ////
    bool addResourcePath(const std::string& path);
    bool addResourceSubPath(const std::string& subPath);

    /// Retrieve Functions
    /** @returns the name of the stored Class this Type loads Resources for */
    const std::string& storedClassName() const { return _typeName; };
    /** @returns the ID of the Type != ClassID */
    /** @returns the type-specific paths this Resource searches in. */
    const std::vector<Directory>& resourcePaths() const { return _resourcePaths; };
    /** @returns the Type specific SubPaths this Resource Searches in @see std::vector<std::string>  _resourceSubPaths */
    const std::vector<Directory>& resourceSubPaths() const { return _resourceSubPaths; };
    /** @returns the Pointers to the Stored resources. @note do not use this, for more than some lookup */
    const std::vector<Resources::StorePointer*>& storedResources() const { return _storedResources; };

    ///////////////////////////////
    //// LOADING AND UNLOADING ////
    virtual void createFromString(const std::string& loadString, const KeepLevel& keepLevel = KeepLevel()) = 0;
    void unloadAllBelowKeepLevel(const Resources::KeepLevel& keepLevel);

    ///////////////////
    //// INTERNALS ////
    void addResource(Resources::StorePointer* resource);

    ///////////////
    //// DEBUG ////
    void debug() const;

  protected:
    Type(const std::string& typeName);

  private:
    Type(const Type& type) {};
  private:
    const std::string                     _typeName;          //!< Name of the Type. (Name of the Resource this loads.)
    std::vector<Directory>                _resourcePaths;     //!< The Paths to search for files in this type
    std::vector<Directory>                _resourceSubPaths;  //!< The subpaths that will be searched under all the _resourcePaths.
    std::vector<std::string>              _fileExtensions;    //!< File Extensions, this Resource supports.

    std::vector<Resources::StorePointer*> _storedResources;   //!< An array of all the stored Resources.
  };

  /**
   * @brief A Type Definition Class for any Object that is resourceable.
   *
   * This Class's main reason of Existence is, that resources can be dynamically
   * created over a loadString. For this the Type of Resource is required, and the Resource must
   * itself support the 'void createFromString(const std::string&)' function.
   */
  template<class T> class tType : public Type
  {
  public:
    /** Create the ResourceType @see Type(const std::string&) */
    tType(const std::string& typeName) : Type(typeName) {};
    /** @param loadString the String to load a Resource with @brief tries to create a Resource of Type T with a loadString */
    virtual void createFromString(const std::string& loadString, const KeepLevel& keepLevel = KeepLevel()) { T::createFromString(loadString, keepLevel); }
  };



  /////////////////////
  // RESOURCE ITSELF //
  /////////////////////
  //! A Resource is an Object, that can be loaded from Disk
  /**
   * The Resource Hanldes the location and stores pointers to data that can be retrieved.
   */
  class Resource : virtual public BaseObject
  {
    ObjectListDeclaration(Resource);

  public:
    Resource(Resources::Type* type);
    virtual ~Resource();

    /** @brief reloads the underlying resource */
    virtual bool reload() { return false; };
    /** @brief unloads the underlying Resource */
    virtual bool unload() { return false; };

    std::string locateFile(const std::string& fileName) const;

  protected:
    Resources::StorePointer* acquireResource(const std::string& loadString);
    void addResource(Resources::StorePointer* pointer);

  private:
    std::string locateFileInSubDir(const Directory& directory, const std::string& fileName) const;

  private:
    Resources::StorePointer*       _pointer;                         //!< Virtual Pointer to the ResourceData.
    Resources::Type*               _type;                            //!< Type of the Resource.
  };
}

#endif /* _RESOURCE_H */
