/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

/*!
 * @file load_param.h
 * A Class and macro-functions, that makes our lives easy to load-in parameters
 */

#ifndef _LOAD_PARAM_H
#define _LOAD_PARAM_H

#include "util/executor/executor_substring.h"
#include "util/executor/executor_member.h"
#include "util/executor/functor_member.h"

#include "parser/tinyxml/tinyxml.h"

// Forward Declaration //
class LoadClassDescription;
class LoadParamDescription;
class TiXmlElement;

/**
 * Loads a Parameter from ROOT named PARAMETER_NAME
 * onto OBJECT of CLASS, trough FUNCTION
 * @param ROOT the TiXmlElement to load the Parameter from
 * @param PARAMETER_NAME the Name of the Parameter to load
 * @param OBJECT The BaseObject to load the new setting to.
 * @param CLASS What Class the BaseObejct is of (this is for identifying the Functuon)
 * @param FUNCTION The function of Class to Load (if you want to call &CLASS::FUNCTION write FUNCTION here).
 */
#define LoadParam(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION) \
         CLoadParam<CLASS>(ROOT, PARAMETER_NAME, OBJECT, createExecutor<CLASS, CLASS>(&CLASS::FUNCTION), false)

/**
 * @brief Does essentially the same as LoadParam, but within a Cycle in an ordered fashion.
 *
 * This Function looks in each Element, if the PARAMETER_NAME matches, and loads onto OBJECT
 * of CLASS the ROOT through FUNCTION
 *
 * @see LoadParam(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION)
 */
#define LoadParam_CYCLE(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION) \
         CLoadParam<CLASS>(ROOT, PARAMETER_NAME, OBJECT, createExecutor<CLASS, CLASS>(&CLASS::FUNCTION), true)

/**
 * this Starts a Cycle in the Loading Process
 * be aware, that in the cycle the first parameter of load_param should because
 * called element, and that you must say true at the Fith parameter, or it will fail
 * also you will have to close the Cycle again with LOAD_PARAM_END_CYCLE
 *
 * @param ROOT The root XLM-element to search element under.
 * @param ELEMENT the element to search
 */
#define LOAD_PARAM_START_CYCLE(ROOT, ELEMENT) \
  const TiXmlElement* ELEMENT; \
  ELEMENT= ROOT->FirstChildElement(); \
  while( ELEMENT != NULL) \
{
/**
   * closes a LoadParam Loop
   * @see LOAD_PARAM_START_CYCLE
   * @param ELEMENT the Element to step through.
 */
#define LOAD_PARAM_END_CYCLE(ELEMENT) \
  ELEMENT = ELEMENT->NextSiblingElement(); \
}


/**************************
**** REAL DECLARATIONS ****
**************************/
//!< A BaseClass for all LoadParam's.
class LoadParamBase
{
protected:
  LoadParamBase(const TiXmlElement* root, const std::string& paramName, bool inLoadCycle = false);

protected:
  void describe(const ClassID& classID, const std::string& descriptionText);
  void setDescriptionValues(const ClassID& classID, unsigned int paramCount, const MultiType* const defaultValues, bool retVal = false);

public:
  static std::string grabParameter(const TiXmlElement* root, const std::string& parameterName);
  static const TiXmlElement* grabParameterElement(const TiXmlElement* root, const std::string& parameterName);

protected:
  const std::string        paramName;            //!< The Name of the Parameter this LoadParams applies to.
  bool                     inLoadCycle;          //!< If the Parameter is in a LoadCycle.

  const TiXmlElement*      loadElem;             //!< The Element to load.
};


//! The Loading Class of the LoadParam, that acctually executes the loading process.
template <class OperateClass> class CLoadParam : public LoadParamBase
{
public:
  /**
   * @brief generates a LoadParam based on:
   * @param root the Root Element to load onto the object. @param paramName the Parameter name that is loaded.
   * @param object the Object to apply the changes on. @param executor the Functional Object, that actually executes the function Call.
   * @param inLoadCycle If we are inside of a loading cycle. (Loading will be different here)
   */
  CLoadParam(const TiXmlElement* root, const std::string& paramName, OperateClass* object, Executor<const SubString, OperateClass>* executor, bool inLoadCycle = false)
      : LoadParamBase(root, paramName, inLoadCycle)
  {
    assert (executor != NULL);
    this->object = object;
    this->executor = executor;
  }
  virtual ~CLoadParam()
  {
    std::string loadString;
    if (this->loadElem != NULL &&  this->loadElem->ToText())
    {
      loadString = this->loadElem->Value();
      if (!loadString.empty())
      {
        /*	PRINTF(4)("Loading value '%s' with Parameters '%s' onto: %s::%s\n",
        	this->paramName.c_str(), loadString.c_str(), this->object->getClassCName(), this->object->getCName());*/
        (*this->executor)(this->object, SubString(loadString, ",", SubString::WhiteSpaces, false, '\\'));
      }
    }
    this->setDescriptionValues(OperateClass::staticClassID(), executor->getParamCount(), executor->getDefaultValues(), executor->hasRetVal());
    delete this->executor;
  }
  /**
   * @brief set the default values of the executor
   * @param value0 the first default value   @param value1 the second default value
   * @param value2 the third default value   @param value3 the fourth default value
   * @param value4 the fifth default value
   */
  CLoadParam& defaultValues(const MultiType& value0 = MT_NULL, const MultiType& value1 = MT_NULL,
                            const MultiType& value2 = MT_NULL, const MultiType& value3 = MT_NULL,
                            const MultiType& value4 = MT_NULL)
  { this->executor->defaultValues(value0, value1, value2, value3, value4); return *this;  };
  //! Describes a LoadParam
  CLoadParam& describe(const std::string& descriptionText) { LoadParamBase::describe(OperateClass::staticClassID(), descriptionText); return *this; };
  //     CLoadParam& attribute(const std::string& attributeName, const Executor<SubString>& executor);

private:
  Executor<const SubString, OperateClass>*         executor;            //!< The Executor, that actually executes the Loading process.
  OperateClass*                                    object;              //!< The Object this LoadParam operates on.
};

#endif /* _LOAD_PARAM_H */
