/*!
 * @file executor_xml.h
 * Definition of a on-screen-shell
 */

#ifndef _EXECUTOR_XML_H
#define _EXECUTOR_XML_H

#include "executor.h"
#include "compiler.h"
#include "debug.h"

#include "parser/tinyxml/tinyxml.h"

//! Executes a Function with a const TiXmlElement* parameter.
/**
 * This is a Special Executor, that finds ParamName in root as XML-sub-element
 * This is especially usefull, if you have to Load a SubElement from root in a new Function
 * What must be defined is a XML-root to search the ParameterName under an  a Function to call.
 */
template<class T, class BaseClass = BaseObject> class ExecutorXML : public Executor<const TiXmlElement*>
{
public:
  /**
   * @brief Constructor of a ExecutorXML
   * @param function a Function to call
   * @param root The XML root to search paramName under
   * @param paramName the ParameterName the search in root, and lookup the TiXmlElement from
   */
  ExecutorXML(void(T::*function)(const TiXmlElement*), const TiXmlElement* root, const std::string& paramName)
      : Executor<const TiXmlElement*>(false, MT_EXT1)
  {
    PRINTF(4)("Loading %s from XML-element %p\n", paramName.c_str(), root);

    if (likely(root != NULL))
      this->element = root->FirstChildElement(paramName);
    else
      this->element = NULL;

    this->paramName = paramName;
    this->functionPointer = function;
  }

  /**
   * @brief clones an ExecutorXML, used to copy this Element.
   * @returns a _new_ Copy of this Executor
   */
  virtual Executor<const TiXmlElement*>* clone () const
  {
    return new ExecutorXML<T>(functionPointer, element, paramName);
  }

  /**
   * @brief executes the Command on BaseObject
   * @param object the BaseObject to execute this Executor on
   * @param element ignored in this case
   */
  virtual void operator()(BaseObject* object, const TiXmlElement*& element) const
  {
    assert (object != NULL);
    if (this->element != NULL)
      (dynamic_cast<T*>(object)->*(functionPointer))(this->element);
  }

private:
  void    (T::*functionPointer)(const TiXmlElement*);  //!< The functionPointer to the function to be called.
  const   TiXmlElement* element;                       //!< The XML-element to call.
  std::string           paramName;                     //!< The Name of the Parameter this Executor should call.
};

#endif /* _EXECUTOR_XML_H */
