/*!
 * @file debug_buffer.h
 * @brief The Debug buffer Tasks
 * @see debug.h
*/

#ifndef _DEBUG_BUFFER_H
#define _DEBUG_BUFFER_H

#include <string>
#include <list>

#define      DEBUG_BUFFER_SIZE       16384         //!< The Size of the input-buffers (should be large enough to carry any kind of input)

  //! A class handling output from orxonox via debug.h
  /**
   * the DebugBuffer redirects output from PRINTF(x) to the Debug and STDOUT
   * the DebugBuffer is a front-filling queue of limited length, that has the
   * youngest added Entry at the beginning, and the oldest at the end.
   */
  class DebugBuffer
  {
  public:
    virtual ~DebugBuffer();
    /** @returns a Pointer to the only object of this Class */
    inline static DebugBuffer* getInstance() { if (!DebugBuffer::singletonRef) DebugBuffer::singletonRef = new DebugBuffer();  return DebugBuffer::singletonRef; };
    /** @returns true if this class is instanciated, false otherwise */
    inline static bool isInstanciated() { return (DebugBuffer::singletonRef == NULL)?false:true; };

    static void addBufferLineStatic(const char* line, ...);
    void addBufferLine(const char* line);

    /// BUFFER
    /** @param bufferSize the new Buffer-Size */
    void setMaxBufferSize(unsigned int maxBufferSize) { this->maxBufferSize = maxBufferSize; };
    void flush();

    /** @returns the List of stings from the Buffer */
    const std::list<std::string>& getBuffer() const { return this->buffer; };
    /** @returns the Count of lines processed by the Debug. */
    inline unsigned long getLineCount() const { return this->lineCount; };
    /** @returns the Current Buffer Size. */
    inline unsigned int getBufferSize() const { return this->buffer.size(); };

    void debug() const;

  private:
    DebugBuffer();

  private:
    static DebugBuffer*           singletonRef;                       //!< The singleton-reference to the only memeber of this class.
    unsigned int                  maxBufferSize;                      //!< The Size of the buffer

    std::string                   keepBuffer;                         //!< a BUFFER to have multi-non-newLine commands be copied into the debug.

    unsigned long                 lineCount;                          //!< how many Lines have been written out so far.

    // The Beginning of buffer (buffer.front()) is the last added line.
    static char                   bufferArray[DEBUG_BUFFER_SIZE];     //!< a BUFFER for fast writing
    static std::list<std::string> buffer;                             //!< A list of stored char-arrays(strings) to store the history
  };

#endif /* _DEBUG_BUFFER_H */
