/*!
 * @file color.h
 * @brief Definition of color-calculations
 *
 * TODO CONVERT THE VECTORS to COLORS!!
 *
 * code borrowed from:
 * http://www.easyrgb.com/math.php
*/

#ifndef _COLOR_H
#define _COLOR_H

#include "vector.h"

//! A Class that handles Colors.
/**
 * A Color is a collection of 4 values:
 * <ul>
 *  <li>Red</li>
 *  <li>Green</li>
 *  <li>Blue</li>
 *  <li>Alpha</li>
 * </ul>
 * With these four values any color of the entire spectrum can be defined.
 *
 * By default a Color lies between 0 and 1 for each component.
 * for example [1,0,0,.5] means red and half visible.
 */
class Color
{
public:
  /** @param r red, @param g green @param b blue @param a alpha @brief constructs a Color. */
  Color(float r = 0.0f, float g = 0.0f, float b = 0.0f, float a = 1.0f) { _rgba[0] = r; _rgba[1] = g; _rgba[2] = b; _rgba[3] = a; };
  /** @param c Color @brief copy constructor */
  Color(const Color& c) { _rgba[0] = c.r(); _rgba[1] = c.g(); _rgba[2] = c.b(); _rgba[3] = c.a(); }

  /** @param c the Color to set to this color @returns the copied color */
  inline const Color& operator=(const Color& c) { _rgba[0] = c.r(); _rgba[1] = c.g(); _rgba[2] = c.b(); _rgba[3] = c.a(); return *this; };
  /** @param c the color to compare @returns true on match. @brief compares two colors */
  inline bool operator==(const Color& c) const { return (r() == c.r() && g() == c.g() && b() == c.b() && a() == c.a()); };

  /** @returns the i'th Value of the Color @param i part of the color 0:r, 1:g, 2:b, 3:a */
  inline float& operator[](unsigned int i) { return _rgba[i]; }
  /** @returns a Constant Value of the color. @param i part of the color 0:r, 1:g, 2:b, 3:a */
  inline const float& operator[](unsigned int i) const { return _rgba[i]; }

  /** @returns the red part. */
  inline float r() const { return _rgba[0]; }
  /** @returns the reference to the red part */
  inline float& r() { return _rgba[0]; }
  /** @returns the green part. */
  inline float g() const { return _rgba[1]; }
  /** @returns the reference to the green part */
  inline float& g() { return _rgba[1]; }
  /** @returns the blue part */
  inline float b() const { return _rgba[2]; }
  /** @returns the reference to the blue part */
  inline float& b() { return _rgba[2]; }
  /** @returns the alpha part */
  inline float a() const { return _rgba[3]; }
  /** @returns the reference to the alpha part */
  inline float& a() { return _rgba[3]; }


  /** @param r red, @param g green @param b blue @param a alpha @brief sets the color. */
  void setColor(float r = 0.0f, float g = 0.0f, float b = 0.0f, float a = 1.0f) { _rgba[0] = r; _rgba[1] = g; _rgba[2] = b; _rgba[3] = a; };
  /** @param c the color to set. @brief sets the color. */
  void setColor(const Color& c) { r() = c.r();  g()= c.g(); b() = c.b(); a() = c.a(); };

  /** @returns the distance to the color @param c the color to calculate the distance to. */
  inline float dist(const Color& c) const { return (sqrt((r()-c.r())*(r()-c.r()) + (g()-c.g())*(g()-c.g()) + (b()-c.b())*(b()-c.b()) + (a()-c.a())*(a()-c.a()))); }
  /// Maths
  /** @param c the color to add to this one @returns the two added colors */
  inline const Color& operator+=(const Color& c) { r()+=c.r(); g()+=c.g(); b()+=c.b(); a()+=c.a(); return *this; };
  /** @returns the result of the added colors @param c the color to add */
  inline Color operator+(const Color& c) const { return Color(r()+c.r(), g()+c.g(), b()+c.b(), a()+c.a()); };
  /** @param c the color to substract to this one @returns the two substracted colors */
  inline const Color& operator-=(const Color& c) { r()-=c.r(); g()-=c.g(); b()-=c.b(); a()-=c.a(); return *this; };
  /** @returns the result of the substracted colors @param c the color to substract */
  inline Color operator-(const Color& c) const { return Color(r()-c.r(), g()-c.g(), b()-c.b(), a()-c.a()); };
  /** @param v the multiplier @returns the Color multiplied by v */
  inline const Color& operator*=(float v) { r()*=v, g()*=v, b()*=v, a()*=v; return *this; };
  /** @param v the multiplier @returns a multiplied color */
  inline Color operator*(float v) const { return Color(r()*v, g()*v, b()*v, a()*v); };

  /** @param c the color to slerp to @param v how much to slerp [0:1] @brief moves the color into the direction of another color */
  void slerp(const Color& c, float v) { *this += (c - *this) * v; };
  void slerpHSV(const Color& c, float v);
  static Color slerpHSVColor(const Color& from, const Color& to, float v);

  void debug() const;

  /// STATIC TRANSFORMATIONS
public:
  static Vector RGBtoHSV (const Vector& RGB);
  static void RGBtoHSV (const Vector& RGB, Vector& HSV);
  static Vector HSVtoRGB (const Vector& HSV);
  static void HSVtoRGB (const Vector& HSV, Vector& RGB);

private:
  static float minrgb(float r, float g, float b);
  static float maxrgb(float r, float g, float b);

public:
  static const Color red;
  static const Color green;
  static const Color blue;
  static const Color white;
  static const Color black;

private:
  float       _rgba[4]; //!< Color Values [r,g,b,a] (red green blue alpha)
};

#endif /* _COLOR_H */
