/*!
 * @file resource_manager.h
 */

#ifndef _RESOURCE_MANAGER_H
#define _RESOURCE_MANAGER_H

#include "resource.h"
#include "filesys/file.h"
#include "filesys/directory.h"

namespace Resources
{

  class ResourceManager : public BaseObject
  {
    ObjectListDeclaration(ResourceManager);
  public:
    ///////////////////////
    //// INSTANZIATION ////
    /** @returns a Pointer to the only object of this Class */
    inline static ResourceManager* getInstance() { if (!_singletonRef) _singletonRef = new ResourceManager();  return _singletonRef; };
    /** @brief deletes the Instance if it exists. */
    inline static void deleteInstance() { if (_singletonRef) delete _singletonRef; };

    ////////////////////////
    //// RESOURCE PATHS ////
    void setMainGlobalPath(const Directory& directory);
    void addGlobalPath(const Directory& directory);

    bool addResourcePath(const std::string& resourceName, const std::string& pathName);
    bool addResourceSubPath(const std::string& resourceName, const std::string& pathName);
    void registerType(Resources::Type* type);
    void unregisterType(Resources::Type* type);

    /** @returns the main global search Path */
    const Directory& mainGlobalPath() const { return _mainGlobalPath; };
    /** @returns all global paths without mainGlobalPath */
    const std::vector<Directory>& globalPaths() const { return _globalPaths; };

    ////////////////////
    //// KEEPLEVELS ////
    unsigned int addKeepLevelName(const std::string& keepLevelName);
    unsigned int getKeepLevelID(const std::string& keepLevelName) const;
    const std::string& getKeepLevelName(unsigned int keepLevelID) const;
    void setDefaultKeepLevel(const KeepLevel& keepLevel) { this->_defaultKeepLevel = keepLevel; };
    const KeepLevel& defaultKeepLevel() const { return this->_defaultKeepLevel; };

    //////////////////////////
    //// GENERAL QUERIES ////
    /** @returns the Types of Resources */
    const std::vector<Resources::Type*> resourceTypes() const { return _resourceTypes; };

    bool checkFileInMainPath(const File& fileInside);
    std::string prependAbsoluteMainPath(const std::string& fileName);

    ////////////////////////////////////////
    //// RESOURCE LOADING AND UNLOADING ////
    void loadFromLoadString(const std::string& resourceTypeName, const std::string& loadString, const KeepLevel& keepLevel = KeepLevel());
    void loadFromLoadStringHACK(const std::string& resourceTypeName, const std::string& loadString) { this->loadFromLoadString(resourceTypeName, loadString); };

    void unloadAllBelowKeepLevel(const Resources::KeepLevel& keepLevel);
    void unloadAllBelowKeepLevelINT(unsigned int level) { unloadAllBelowKeepLevel(level); };

    ///////////////
    //// DEBUG ////
    void debug() const;

  private:
    ResourceManager();
    virtual ~ResourceManager();

  private:
    static ResourceManager*            _singletonRef;       //!< singleton Reference

    Directory                          _mainGlobalPath;     //!< The main include directory (default at "./")
    std::vector<Directory>             _globalPaths;        //!< Additional Global include directories.

    std::vector<Resources::Type*>      _resourceTypes;      //!< A Vector of all the stored ResourceTypes @see Resources::Type

    std::vector<std::string>           _keepLevelNames;     //!< Names of KeepLevels @see Resources::KeepLevel
    KeepLevel                          _defaultKeepLevel;   //!< The default KeepLevel.
  };

}

#endif /* _RESOURCE_MANAGER_H */
