/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

/*!
 * @file load_param_xml.h
 * A Class and macro-functions, that makes our lives easy to load-in parameters from XML
 */

#ifndef _LOAD_PARAM_XML_H
#define _LOAD_PARAM_XML_H

#include "load_param.h"
#include "executor/executor_xml.h"

/**
 * @brief Loads a Parameter from ROOT named PARAMETER_NAME
 * onto OBJECT of CLASS, trough FUNCTION
 * @param ROOT the TiXmlElement to load the Parameter from
 * @param PARAMETER_NAME the Name of the Parameter to load
 * @param OBJECT The BaseObject to load the new setting to.
 * @param CLASS What Class the BaseObejct is of (this is for identifying the Functuon)
 * @param FUNCTION The function of Class to Load (if you want to call &CLASS::FUNCTION write FUNCTION here).
 */
#define LoadParamXML(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION) \
         XmlLoadParam<CLASS>(ROOT, PARAMETER_NAME, OBJECT, new ExecutorXML<CLASS, CLASS>(&CLASS::FUNCTION, ROOT, PARAMETER_NAME), false)


/**
 * @brief Does essentially the same as LoadParamXML, but within a Cycle in an ordered fashion.
 *
 * This Function looks in each Element, if the PARAMETER_NAME matches, and loads onto OBJECT
 * of CLASS the ROOT through FUNCTION
 *
 * @see LoadParamXML(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION)
 */
#define LoadParamXML_CYCLE(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION) \
         XmlLoadParam<CLASS>(ROOT, PARAMETER_NAME, OBJECT, new ExecutorXML<CLASS, CLASS>(&CLASS::FUNCTION, ROOT, PARAMETER_NAME), true)

//! A Class that can load XML tags onto an Object.
template <class OperateClass = BaseObject> class XmlLoadParam : public LoadParamBase
{
public:
  /**
   * @brief generates a LoadParam based on
   * @param root the Root Element to load onto the object.
   * @param paramName the Parameter name that is loaded.
   * @param object the Object to apply the changes on.
   * @param executor the Functional Object, that actually executes the function Call.
   * @param inLoadCycle If we are inside of a loading cycle. (Loading will be different here)
   */
  XmlLoadParam(const TiXmlElement* root, const std::string& paramName, OperateClass* object, Executor<const TiXmlElement*>* executor, bool inLoadCycle = false)
      : LoadParamBase(root, paramName, inLoadCycle), object(object) , executor(executor)
  {
    assert(this->object != NULL);
    assert(this->executor != NULL);
  }

  /** @brief executes and destroys the executor */
  virtual ~XmlLoadParam()
  {
    this->setDescriptionValues(OperateClass::staticClassID(), executor->getParamCount(), executor->getDefaultValues(), executor->hasRetVal());
    (*this->executor)(this->object, this->loadElem);
    delete this->executor;
  }

  /** @param descriptionText the description @returns self @brief describes the Loading parameter. */
  XmlLoadParam& describe(const std::string& descriptionText) { LoadParamBase::describe(OperateClass::staticClassID(), descriptionText); return *this; };

private:
  OperateClass*                        object;         //!< The Object to apply this to.
  Executor<const TiXmlElement*>*    executor;       //!< The Executor, that does the actual loading process.
};

#endif /* _LOAD_PARAM_XML_H */
