/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#include "load_param_description.h"

#include "multi_type.h"
#include "debug.h"

/**
 * @brief Creates a Description of a LoadParam
 * @param paramName the name of the parameter to load
 */
LoadParamDescription::LoadParamDescription(const std::string& paramName)
    : _name(paramName), _parameterCount(0)
{ }

/**
 * @param descriptionText The text to set as a description for this Parameter
 */
void LoadParamDescription::setDescription(const std::string& descriptionText)
{
  this->_description = descriptionText;
}

/**
 * @brief sets the Values of the LoadParam in the Description.
 * @param paramCount the count of arguments the underlying paramDescription takes.
 * @param defaultValues the default Values the underlying parameter takes.
 * @param retVal if the underlying parameter has a return value
 */
void LoadParamDescription::setValues(unsigned int paramCount,
                                     const MultiType* const defaultValues,
                                     bool retVal)
{
  this->_parameterCount = paramCount;
  for (unsigned int i = 0; i < paramCount; ++i)
  {
    this->_defaultValues.push_back(defaultValues[i].getString());
    this->_types.push_back(MultiType::MultiTypeToString(defaultValues[i].getType()));
  }

}


/**
 * @brief prints out this parameter, its input method and the description (if availiable)
 * @param stream the stream to print to.
 * @param withComments if the comments should be appended.
 */
void LoadParamDescription::print(FILE* stream, bool withComments) const
{
  fprintf(stream, " <%s>", this->_name.c_str());
  for (unsigned int i = 0; i < this->_parameterCount; i++)
  {
    if (i > 0)
      fprintf(stream, ",");
    fprintf(stream, "%s", this->_types[i].c_str());
  }
  fprintf(stream, "</%s>", this->_name.c_str());
  // here the comments are printed out.
  if (withComments)
  {
    if (!this->_description.empty())
      fprintf(stream, " <!-- %s", this->_description.c_str());
    // default values
    if (this->_parameterCount > 0)
    {
      fprintf(stream, " (Default: ");
      for (unsigned int i = 0; i < this->_parameterCount; i++)
      {
        if (i > 0)
          fprintf(stream, ", ");
        if (this->_types[i] == "string")
        { // leave brackets !!
          fprintf(stream, "\"%s\"", this->_defaultValues[i].c_str());
        }
        else
        {
          fprintf(stream, "%s", this->_defaultValues[i].c_str());
        }
      }
      fprintf(stream, ")");
    }
    if (!this->_description.empty() || this->_parameterCount > 0)
      fprintf(stream, " -->");
  }
  fprintf(stream, "\n");
}


