/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#include "load_param_class_description.h"

#include "multi_type.h"
#include "debug.h"
#include <cassert>

/**
 * @brief A list, that holds all the classes that are loadable (classes not objects!!)
 */
LoadParamClassDescription::ClassDescriptionMap LoadParamClassDescription::_classList;

/**
 * @brief if the description of Parameters should be executed
 */
bool LoadParamClassDescription::_captureDescriptions = false;

/**
 * @param className the name of the class to be loadable
 */
LoadParamClassDescription::LoadParamClassDescription(const std::string& className)
    : _className(className)
{ }


/**
 * @brief clears all LoadParamDescriptions.
 */
void LoadParamClassDescription::deleteAllDescriptions()
{
  LoadParamClassDescription::_classList.clear();
}


/**
 * @brief describes a LoadParam-parameter.
 * @param classID the ID of the class.
 * @param paramName the Name of the Parameter.
 * @param descriptionText the Test to set.
 */
void LoadParamClassDescription::describeClass(const ClassID& classID,
    const std::string& paramName,
    const std::string& descriptionText)
{
  ParamDescriptionMap::iterator it = LoadParamClassDescription::getParamDescription(classID, paramName);

  (*it).second.setDescription(descriptionText);
}

/**
 * @brief sets Values of the specified LoadParam-parameter.
 * @param classID the ID of the class.
 * @param paramName the name of the Parameter.
 * @param paramCount the count of Parameters this LoadParam takes.
 * @param defaultValues the default Values.
 * @param retVal if the Parameter takes return Values.
 */
void LoadParamClassDescription::setValuesOf(const ClassID& classID,
    const std::string& paramName,
    unsigned int paramCount,
    const MultiType* const defaultValues,
    bool retVal)
{
  ParamDescriptionMap::iterator it = LoadParamClassDescription::getParamDescription(classID, paramName);
  (*it).second.setValues(paramCount, defaultValues, retVal);
}


/**
 * @brief finds the Iterator to the ParameterDescription paramName matching classID
 * @param classID the ClassID to match.
 * @param paramName the name of the parameter in the Class.
 * @returns the iterator on match.
 *
 * @note this function creates the Element classID.name()::paramName on the go if it does not exist.
 */
LoadParamClassDescription::ParamDescriptionMap::iterator
LoadParamClassDescription::getParamDescription(const ClassID& classID, const std::string& paramName)
{
  /// Locate the ClassDescription first
  ClassDescriptionMap::iterator classIt = LoadParamClassDescription::_classList.find(classID);
  if (classIt == LoadParamClassDescription::_classList.end())
  {
    LoadParamClassDescription::_classList[classID] = LoadParamClassDescription(classID.name());
    classIt = LoadParamClassDescription::_classList.find(classID);
  }
  // At this position the class-iterator should point to a valid usefull position.
  assert(classIt != LoadParamClassDescription::_classList.end());

  /// Now locate the description with paramName.
  ParamDescriptionMap::iterator paramIt = (*classIt).second._parameters.find(paramName);
  if (paramIt == (*classIt).second._parameters.end())
  {
    (*classIt).second._parameters[paramName] = LoadParamDescription(paramName);
    paramIt = (*classIt).second._parameters.find(paramName);
  }
  // at this position the param-iterator should
  assert (paramIt != (*classIt).second._parameters.end());

  return (paramIt);
}

/**
 * @brief prints out a nice output about this Classes Loading, and their parameters
 * @param stream the stream to write to.
 * @param withComments if Comments should be included.
 */
void LoadParamClassDescription::print(FILE* stream, bool withComments) const
{
  fprintf(stream, "<%s>\n", this->_className.c_str());
  for (ParamDescriptionMap::const_iterator param = this->_parameters.begin();
       param != this->_parameters.end();
       ++param)
  {
    (*param).second.print(stream, withComments);
  }
  fprintf(stream, "</%s>\n\n", this->_className.c_str());
}


/**
 * @brief prints out all loadable Classes, and their parameters
 * @param fileName prints the output to a File
 * @param withComments if Comments should be included.
 */
void LoadParamClassDescription::printAll(const std::string& fileName, bool withComments)
{
  FILE* stream;
  if( (stream = fopen (fileName.c_str(), "w")) == NULL)
  {
    PRINTF(2)("File '%s' could not be opened for writing\n Printing to stdout\n", fileName.c_str());
    LoadParamClassDescription::printAllTo(stdout, withComments);
    return;
  }
  else
    LoadParamClassDescription::printAllTo(stream, withComments);

  fclose (stream);
}

/**
 * @brief prints out all loadable Classes, and their parameters to a stream
 * @param stream the stream to print to.
 * @param withComments if Comments should be included.
 */
void LoadParamClassDescription::printAllTo(FILE* stream, bool withComments)
{
  fprintf(stream, "===============================================================\n");
  fprintf(stream, " Listing all the Loadable Options (loaded since Game started).\n\n");
  for (ClassDescriptionMap::const_iterator classIt = LoadParamClassDescription::_classList.begin();
       classIt != LoadParamClassDescription::_classList.end();
       classIt ++)
  {
    (*classIt).second.print(stream, withComments);
  }
  fprintf(stream, "===============================================================\n");
}
