/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_LOADING

#include "load_param.h"
#include "load_param_class_description.h"
#include "compiler.h"
#include "debug.h"
/**
 * @brief Constructs a new LoadParameter
 * @param root the XML-element to load this Parameter from
 * @param paramName the Parameter to load
 * @param inLoadCycle If we are in a LoadCycle (loading differs.).
 */
LoadParamBase::LoadParamBase(const TiXmlElement* root, const std::string& paramName, bool inLoadCycle)
    :  paramName(paramName), inLoadCycle(inLoadCycle)
{
  // determin the LoadString.
  if (likely(!inLoadCycle))
    this->loadElem = grabParameterElement(root, paramName);
  else if (paramName == root->Value())
    this->loadElem = (TiXmlElement*)root->FirstChild();
  else
    this->loadElem = NULL;
}


/**
 * @param classID the ID of the class. This is needed to identify into what class this Parameter belongs.
 * @param descriptionText The text to set as a description for this Parameter
 * @returns a pointer to itself.
 */
void LoadParamBase::describe(const ClassID& classID, const std::string& descriptionText)
{
  PRINTF(5)("Describing Class '%s'(id:%d) Parameter '%s': description '%s'\n",
            classID.name().c_str(), classID.id(), paramName.c_str(), descriptionText.c_str());

  if (LoadParamClassDescription::descriptionsCaptured())
    LoadParamClassDescription::describeClass(classID, paramName, descriptionText);
}

/**
 * @brief sets the Values of the Description to a usefull text.
 */
void LoadParamBase::setDescriptionValues(const ClassID& classID, unsigned int paramCount, const MultiType* const defaultValues, bool retVal)
{
  if(LoadParamClassDescription::descriptionsCaptured())
    LoadParamClassDescription::setValuesOf(classID, paramName, paramCount, defaultValues, retVal);
}


//////////////////////
// HELPER FUNCTIONS //
//////////////////////
/**
 * @param root: The XML-element to grab a parameter from
 * @param parameterName: the parameter to grab
 * @returns the Value of the parameter if found, NULL otherwise
*/
std::string LoadParamBase::grabParameter(const TiXmlElement* root, const std::string& parameterName)
{
  const TiXmlElement* const element = grabParameterElement(root, parameterName);
  if (element != NULL)
    return element->Value();
  else
  {
    static std::string empty("");
    return empty;
  }
}

/**
 * @param root: The XML-element to grab a parameter from
 * @param parameterName: the parameter to grab
 * @returns the Element of the parameter if found, NULL otherwise
 */
const TiXmlElement* LoadParamBase::grabParameterElement(const TiXmlElement* root, const std::string& parameterName)
{
  const TiXmlElement* element;
  const TiXmlNode* node;

  if (root == NULL)
    return NULL;

  element = root->FirstChildElement( parameterName);
  if( element == NULL) return NULL;

  node = element->FirstChild();
  while( node != NULL)
  {
    if( node->ToText()) return (TiXmlElement*)node;
    node = node->NextSibling();
  }
  return NULL;
}
