/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christian Meyer
   co-programmer: Benjamin Grauer
*/

/*!
 * @file factory.h
 * @brief A loadable object handler
*/


#ifndef __FACTORY_H
#define __FACTORY_H

class BaseObject;

#include "parser/tinyxml/tinyxml.h"
#include "base_object.h"
#include <map>

/**
 * Creates a factory to a Loadable Class.
 * this should be used at the beginning of all the Classes that should be loadable (in the cc-file)
 */
#define CREATE_FACTORY(CLASS_NAME) \
    tFactory<CLASS_NAME> global_##CLASS_NAME##_Factory = tFactory<CLASS_NAME>(CLASS_NAME::staticClassID())

//! The Factory is a loadable object handler
class Factory : public BaseObject
{
  //! Declare the ObjectList at the BaseObject List
  ObjectListDeclaration(Factory);
public:
  virtual ~Factory ();

  static  BaseObject* fabricate(const std::string& className);
  static  BaseObject* fabricate(const ClassID& classID);
  static  BaseObject* fabricate(const TiXmlElement* root);

  bool operator==(int classID) const;
  bool operator==(const std::string& className) const;
  /** @param classID the ID to compare @returns true if the ID's match */
  bool operator==(const ClassID& classID) const { return _classID == classID; };


  void debug() const;
  static void debugAll();

protected:
  Factory (const ClassID& id);
  /**
   * @brief The core function of the Factory. Creates objects of the Factories Type.
   * @param root The TiXmlElement of the Factory.
   * @returns the created Object in BaseType* format.
   */
  virtual BaseObject* fabricateObject(const TiXmlElement* root = NULL) const = 0;

private:
  //! Copy Constructor is hidden.
  Factory (const Factory&) {};

protected:
  /** The Type of the FactoryMap that is sorted by ID */
  typedef std::map<const ClassID, Factory*>  FactoryIDMap;
  /** The Type of the FactoryMap that is sorted by Name */
  typedef std::map<std::string, Factory*>    FactoryStringMap;

  const ClassID                 _classID;              //!< The Class-Identifyer of the Factory.
  static FactoryIDMap           _factoryIDMap;         //!< List of Registered Factories
  static FactoryStringMap       _factoryStringMap;     //!< List of Registered Factories
};

/**
 *  @brief a factory that is able to load any kind of Object
 * (this is a Functor)
 */
template<class T> class tFactory : public Factory
{
public:
  /**
   * @brief creates a new type Factory to enable the loading of T
   * @param classID the ID of the Class to be created.
   */
  tFactory (const ClassID& classID)
      : Factory(classID)
  {  }
  /**
   * @brief copy constructor
   * @param factory the Factory to copy
   */
  tFactory (const tFactory& factory) : Factory(factory._classID) {};

  /**
   * @brief fabricates an Object of type T, with the constructor T::T(const TiXmlElemnt*)
   * @param root the TiXmlElement T should load parameters from.
   * @return the newly fabricated T.
   */
  virtual BaseObject* fabricateObject(const TiXmlElement* root = NULL) const
  {
    return new T(root);
  }
};

#endif /* _FACTORY_H */

