/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christian Meyer
   co-programmer: Benjamin Grauer
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_LOADING

#include "factory.h"
#include "debug.h"
//#include "shell_command.h"

ObjectListDefinition(Factory);

//SHELL_COMMAND(create, Factory, fabricate);

/**
 * @brief constructor
 *
 * set everything to zero and define factoryName
 */
Factory::Factory (const ClassID& classID)
    : _classID(classID)
{
  PRINTF(4)("Factory::create(%s::%d)\n", classID.name().c_str(), classID.id());
  //this->registerObject(this, Factory::_objectList);
  this->setName(classID.name());

  Factory::_factoryIDMap[classID] = this;
  Factory::_factoryStringMap[classID.name()] = this;
}

/** @brief A Map of all Factories ordered by ID. */
Factory::FactoryIDMap Factory::_factoryIDMap;

/** @brief A Map of all Factories ordered by Name. */
Factory::FactoryStringMap Factory::_factoryStringMap;

/**
 * @brief destructor
 */
Factory::~Factory ()
{
  FactoryIDMap::iterator it = Factory::_factoryIDMap.find(this->_classID);
  if (it != Factory::_factoryIDMap.end() && (*it).second == this)
    Factory::_factoryIDMap.erase(it);

  FactoryStringMap::iterator stringIt = Factory::_factoryStringMap.find(this->_classID.name());
  if (stringIt != Factory::_factoryStringMap.end() && (*stringIt).second == this)
    Factory::_factoryStringMap.erase(stringIt);
}

/**
 * @param classID match a classID with this classID
 * @returns true on match, false otherwise
 */
bool Factory::operator==(int classID) const
{
  return (this->_classID == classID);
}


/**
 * @brief Compares the Factories Name against a given ClassName
 * @param className the Name of the Class to Query
 * @returns true on match, false otherwise.
 */
bool Factory::operator==(const std::string& className) const
{
  return (this->_classID.name() == className);
}


/**
 * @brief Creates a new Object of type root->Value() (name)
 * @param root the XML-Root to match for the newly created Object
 * @returns a new Object of Type root->Value() on match, NULL otherwise
 */
BaseObject* Factory::fabricate(const TiXmlElement* root)
{
  FactoryStringMap::const_iterator it = Factory::_factoryStringMap.find(root->Value());
  if (it != Factory::_factoryStringMap.end())
  {
    PRINTF(2)("Create a new Object of type %s\n", (*it).second->getCName());
    return (*it).second->fabricateObject(root);
  }
  else
  {
    PRINTF(2)("Could not Fabricate an Object of Class '%s'\n", root->Value());
    return NULL;
  }
}


/**
 * @brief Creates a new Object of type className
 * @param className the ClassName to match for the newly created Object
 * @returns a new Object of Type className on match, NULL otherwise
 */
BaseObject* Factory::fabricate(const std::string& className)
{
  FactoryStringMap::const_iterator it = Factory::_factoryStringMap.find(className);
  if (it != Factory::_factoryStringMap.end())
  {
    PRINTF(2)("Create a new Object of type %s\n", (*it).second->getCName());
    return (*it).second->fabricateObject(NULL);
  }
  else
  {
    PRINTF(2)("Could not Fabricate an Object of Class '%s'\n", className.c_str());
    return NULL;
  }
}

/**
 * @brief Creates a new Object of type classID
 * @param classID the ClassID to match for the newly created Object
 * @returns a new Object of Type classID on match, NULL otherwise
 */
BaseObject* Factory::fabricate(const ClassID& classID)
{
  FactoryIDMap::const_iterator it = Factory::_factoryIDMap.find(classID);
  if (it != Factory::_factoryIDMap.end())
  {
    PRINTF(4)("Create a new Object of type %s\n", (*it).second->getCName());
    return (*it).second->fabricateObject(NULL);
  }
  else
  {
    PRINTF(2)("Could not Fabricate an Object of ClassID '%d'\n", classID.id());
    return NULL;
  }
}


/**
 * @brief print out some nice litte debug information about the Factory.
 */
void Factory::debug() const
{
  PRINTF(0)("Factory of class '%s' with ClassID: %d\n", this->_classID.name().c_str(), this->_classID.id());
}

/**
 * @brief Prints out some nice Debug information about all factories
 */
void Factory::debugAll()
{
  PRINTF(0)("Debugging all %d Factories\n", Factory::_factoryStringMap.size());
  Factory::FactoryStringMap::const_iterator it;
  for (it = Factory::_factoryStringMap.begin(); it != Factory::_factoryStringMap.end(); ++it)
    (*it).second->debug();
}
