/*!
 * @file directory.h
 * @brief Definition of the Directory Handler class
 */


/**
 * Copyright (C) 2002 Bart Vanhauwaert
 *
 * Permission to use, copy, modify, distribute and sell this software
 * for any purpose is hereby granted without fee. This license
 * includes (but is not limited to) standalone compilation or as part
 * of a larger project.
 *
 * This software is provided "as is" without express or implied warranty.
 *
 * For a full statement on warranty and terms and conditions for
 * copying, distribution and modification, please see the comment block
 * at the end of this file.
 *
 * Version 1
 *
 */

#ifndef __DIRECTORY_H_
#define __DIRECTORY_H_

#include "file.h"
#include <vector>

//! A Directory is a Special file, that contains multiple Files.
/**
 * @example Correct usage
 * Directory dir("/var/log");
 * dir.open();
 * if (dir.fileNameInDir("emerge.log"))
 *  { // do stuff; }
 */
class Directory : public File
{
public:
  Directory(const std::string& directoryName = "");
  Directory(const Directory& directory);
  virtual ~Directory();

  virtual bool open();
  virtual bool close();

  bool create();

  Directory operator+(const Directory& dir) const;
  Directory& operator+=(const Directory& dir);
  Directory& operator--();
  Directory& operator--(int);
  Directory parentDir() const;


  /** @returns if the Directory was opened */
  bool isOpen() const { return this->_opened; }
  /** @returns the FileCount (count of files contained in this directory) */
  unsigned int fileCount() const { return _fileNames.size(); };
  /** @returns the FileNames contained inside of the Directory */
  const std::vector<std::string>& fileNames() const { return this->_fileNames; };
  /** @returns the i'th FileName @param fileNumber the fileNumber (must not bigger than fileCount()) */
  const std::string& operator[](unsigned int fileNumber) const { return this->_fileNames[fileNumber]; };
  /** @returns a formated string containing the FileName, prepended with the directory-Name */
  std::string fileNameInDir(unsigned int fileNumber) const { return this->name() + Directory::delimiter + _fileNames[fileNumber]; };
  /** @returns a File pointing to the File @param fileNumber the fileNumber (must not bigger than fileCount()) */
  File getFile(unsigned int fileNumber) const { return File(fileNameInDir(fileNumber)); };

public:
  static const char           delimiter;        //!< A Delimiter (*nix: '/', windows: '\\')

private:
  bool                        _opened;          //!< If the directory was opened.
  std::vector<std::string>    _fileNames;       //!< The List of Files contained in the directory. (will be filled when open was called.)
};

File operator+(const Directory& dir, const File& file);

#endif /* __DIRECTORY_H_ */

/**
 *
 * The "library", above, refers to the collection of software functions
 * and/or data contained in this file, prepared so as to be conveniently
 * compiled and linked with application programs (which use some of those
 * functions and data) to form executables.
 *
 *                             NO WARRANTY
 *
 * 1. BECAUSE THE LIBRARY IS LICENSED FREE OF CHARGE, THERE IS NO
 * WARRANTY FOR THE LIBRARY, TO THE EXTENT PERMITTED BY APPLICABLE LAW.
 * EXCEPT WHEN OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR
 * OTHER PARTIES PROVIDE THE LIBRARY "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE
 * LIBRARY IS WITH YOU.  SHOULD THE LIBRARY PROVE DEFECTIVE, YOU ASSUME
 * THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
 *
 * 2. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
 * WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY
 * AND/OR REDISTRIBUTE THE LIBRARY AS PERMITTED ABOVE, BE LIABLE TO YOU
 * FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE THE
 * LIBRARY (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
 * RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
 * FAILURE OF THE LIBRARY TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
 * SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGES.
 *
 * END OF TERMS AND CONDITIONS
 *
 */

