/*!
 * @file shell.h
 * Definition of a on-screen-shell
 *
 * @todo Buffer Display in different Colors for different debug mode.
 * @todo choose color of the Font and the background.
 */

#ifndef _SHELL_H
#define _SHELL_H

#include "element_2d.h"
#include "event_listener.h"

#include "shell_input.h"
#include "debug_buffer.h"
#include "material.h"

#define         SHELL_DEFAULT_FONT              "fonts/final_frontier.ttf"
#define         SHELL_DEFAULT_TEXT_COLOR        0.0f, 1.0f, 0.0f, 1.0f
#define         SHELL_DEFAULT_BACKGROUND_COLOR  0.0f, 0.0f, 0.0f, 1.0f

// FORWARD DECLARATION
class MultiLineText;
class ShellInput;
class Material;

//! Namespace of the Shell of ORXONOX.
namespace OrxShell
{
  //! A class that is able to redirect all output to a openGL-Shell, and that one can use to input some commands
  /**
   * the major idea is, that all the Output can be redirected to the Shell,
   * and does not have to be displayed to the opening Shell, this is good,
   * for developers using Windows, where all output is otherwise redirected
   * to stdout.txt
   *
   * Furthermore the Shell should enable us, to input some simple commands
   * Each Class can check itself in to the Shell, and listen for commands.
   *
   * more info:
   * @see ShellCommand
   * @see shell_command.h
   * @see shell_buffer.h
   * @see shell_input.h
   * @see shell_completion.h
   */
  class Shell : public Element2D, public EventListener
  {
    ObjectListDeclaration(Shell);
  public:
    Shell();
    virtual ~Shell();

    void activate();
    void deactivate();
    /** @returns true if the Shell is active, false otherwise. */
    inline bool isActive() const { return this->bActive; };

    void setFont(const std::string& fontFile);
    void setTextSize(unsigned int textSize, unsigned int lineSpacing = 1);
    void setTextColor(float r, float g, float b, float a);
    void setBackgroundColor(float r, float g, float b, float a);
    void setBackgroundImage(const std::string& fileName);

    // BUFFERS
    void setBufferDisplaySize(unsigned int bufferDisplaySize);
    void printToDisplayBuffer(const std::string& text);
    void moveDisplayBuffer(int lineCount);

    void flush();

    void clear();

    // EventListener
    virtual void process(const Event &event);
    // Element2D-functions
    virtual void tick(float dt);
    virtual void draw() const;


    void debug() const;

    void testShell() const;
  private:
    void updateResolution(unsigned int width);
    void repositionText();
    void applyTextSettings(Text* text);
    void applySettings();
    // helpers //
    Vector2D calculateLinePosition(unsigned int lineNumber);

  private:
    // GENERAL
    DebugBuffer*                shellBuffer;            //!< The local ShellBuffer.

    bool                        bActive;                //!< If the shell is active.
    unsigned int                shellHeight;            //!< The hight of the Shell in Pixels.
    unsigned int                lineSpacing;            //!< The Spacing between lines.
    unsigned int                textSize;               //!< The size of the text.
    float                       textColor[4];           //!< The text's color [r,g,b,a].
    std::string                 fontFile;               //!< The file containing the font.
    Material                    backgroundMaterial;     //!< A material for the background.

    // HANDLING TEXT INPUT
    ShellInput                  shellInput;             //!< The inputLine of the Shell.
    // BUFFER
    unsigned int                bufferDisplaySize;      //!< The Size of the Display-buffer, in lines (not in characters).
    std::list<MultiLineText*>   bufferText;             //!< A list of stored bufferTexts for the display of the buffer.

    unsigned long               linesProcessed;         //!< How many Lines have been processed.

    std::list<std::string>::const_iterator  bufferIterator;         //!< used to move through and print the Buffer
  };

}

#endif /* _SHELL_H */
