/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christoph Renner
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "preferences.h"
#include "lib/parser/ini_parser/ini_parser.h"
#include "debug.h"

ObjectListDefinition(Preferences);

/**
 * standard constructor
 */
Preferences::Preferences ()
{
  this->registerObject(this, Preferences::_objectList);
   this->setName("Preferences");
   this->fileName = "";
}

/**
 *  the singleton reference to this class
 */
Preferences* Preferences::singletonRef = NULL;

/**
   @brief standard deconstructor
 */
Preferences::~Preferences ()
{
  Preferences::singletonRef = NULL;
}

/**
 * Check if this item exists
 * @param section name of the section
 * @param name name of the item to check
 * @return true if the item exists
 */
bool Preferences::exists( const std::string& section, const std::string& name)
{
  std::list<prefSection>::const_iterator it = data.begin();

  for ( ; it!=data.end(); it++)
  {
    if ( it->sectionName == section )
    {
      std::list<prefItem>::const_iterator it2 = it->items.begin();

      for ( ; it2!=it->items.end(); it2++)
      {
        if ( it2->name == name )
          return true;
      }

      break;
    }
  }

  return false;
}

/**
 * Check if this section exists
 * @param section name of the section
 * @param name name of the item to check
 * @return true if the item exists
 */
bool Preferences::sectionExists( const std::string& section )
{
  std::list<prefSection>::const_iterator it = data.begin();

  for ( ; it!=data.end(); it++)
  {
    if ( it->sectionName == section )
    {
      return true;
    }
  }

  return false;
}

/**
 * Sets the value of an item. Creates it if doesn't exits.
 * @param section name of the section
 * @param name name of the item
 * @param value value
 */
void Preferences::setString(const std::string& section, const std::string& name, const std::string& value, bool dontSetModified)
{
  MultiType t(value);
  setMultiType(section, name, t, dontSetModified);
}

/**
 * Sets the value of an item. Creates it if doesn't exits.
 * @param section name of the section
 * @param name name of the item
 * @param value value
 */
void Preferences::setInt(const std::string& section, const std::string& name, int value, bool dontSetModified)
{
  MultiType t(value);
  setMultiType(section, name, t, dontSetModified);
}

/**
 * Sets the value of an item. Creates it if doesn't exits.
 * @param section name of the section
 * @param name name of the item
 * @param value value
 */
void Preferences::setFloat(const std::string& section, const std::string& name, float value, bool dontSetModified)
{
  MultiType t(value);
  setMultiType(section, name, t, dontSetModified);
}

/**
 * Get the value of an item
 * @param section name of the section
 * @param name name of the item to check
 * @param defaultValue value to return if item doesn't exist
 * @return value of the item if found. defaultValue else
 */
std::string Preferences::getString(const std::string& section, const std::string& name, const std::string& defaultValue)
{
  return getMultiType(section, name, MultiType(defaultValue)).getString();
}

/**
 * Get the value of an item
 * @param section name of the section
 * @param name name of the item to check
 * @param defaultValue value to return if item doesn't exist
 * @return value of the item if found. defaultValue else
 */
int Preferences::getInt(const std::string& section, const std::string& name, int defaultValue)
{
  return getMultiType(section, name, MultiType(defaultValue)).getInt();
}

/**
 * Get the value of an item
 * @param section name of the section
 * @param name name of the item to check
 * @param defaultValue value to return if item doesn't exist
 * @return value of the item if found. defaultValue else
 */
float Preferences::getFloat(const std::string& section, const std::string& name, float defaultValue)
{
  return getMultiType(section, name, MultiType(defaultValue)).getFloat();
}

/**
 * Sets the value of an item. Creates it if doesn't exits.
 * @param section name of the section
 * @param name name of the item
 * @param value value
 */
void Preferences::setMultiType(const std::string& section, const std::string& name, const MultiType& value, bool dontSetModified)
{
  std::list<prefSection>::iterator it = data.begin();

  for ( ; it!=data.end(); it++)
  {
    if ( it->sectionName == section )
    {
      std::list<prefItem>::iterator it2 = it->items.begin();

      for ( ; it2!=it->items.end(); it2++)
      {
        if ( it2->name == name )
        {
          if (!dontSetModified)
            it2->modified = value.getString() != it2->value.getString();

          it2->value = value;

          return;
        }
      }
      prefItem item;
      item.value = value;
      item.modified = !dontSetModified;
      item.name = name;
      it->items.push_back(item);
      return;
    }
  }

  prefItem item;
  item.value = value;
  item.modified = !dontSetModified;
  item.name = name;

  prefSection sec;
  sec.items.push_back(item);
  sec.sectionName = section;
  data.push_back( sec );
}

/**
 * Get the value of an item
 * @param section name of the section
 * @param name name of the item to check
 * @param defaultValue value to return if item doesn't exist
 * @return value of the item if found. defaultValue else
 */
MultiType Preferences::getMultiType(const std::string& section, const std::string& name,const MultiType& defaultValue)
{
  std::list<prefSection>::const_iterator it = data.begin();

  for ( ; it!=data.end(); it++)
  {
    if ( it->sectionName == section )
    {
      std::list<prefItem>::const_iterator it2 = it->items.begin();

      for ( ; it2!=it->items.end(); it2++)
      {
        if ( it2->name == name )
        {
          return it2->value;
        }
      }

      break;
    }
  }

  return defaultValue;
}

void Preferences::setUserIni(const std::string& fileName)
{
  this->fileName = fileName;
}

bool Preferences::save()
{
  if ( this->fileName == "" )
  {
    PRINTF(1)("You must call setUserIni before you can call save()\n");
    return false;
  }
  IniParser iniParser(this->fileName);

  std::list<prefSection>::iterator it = data.begin();
  bool didChanges = false;
  for ( ; it!=data.end(); it++)
  {
    std::list<prefItem>::iterator it2 = it->items.begin();

    for ( ; it2!=it->items.end(); it2++)
    {
      if ( it2->modified )
      {
        iniParser.editVar(it2->name, it2->value.getString(), it->sectionName);
        didChanges = true;
      }
    }
  }
  /// HACK DO WE HAVE TO CHECK THIS??
  //if ( didChanges )
  {
    iniParser.writeFile( this->fileName );
  }

  return true;
}

/**
 * prints out all section with its items and values
 */
void Preferences::debug()
{
  std::list<prefSection>::iterator it = data.begin();

  for ( ; it!=data.end(); it++)
  {
    PRINTF(0)("%s\n", it->sectionName.c_str());
    std::list<prefItem>::iterator it2 = it->items.begin();

    for ( ; it2!=it->items.end(); it2++)
    {
      PRINTF(0)("--> %s = '%s'%s\n", it2->name.c_str(), it2->value.getString().c_str(), ((!it2->modified)?"":" <modified>"));
    }
  }
}

/**
 * list all keys in section
 * @param section section
 * @return list of keys
 */
std::list< std::string > Preferences::listKeys( const std::string section )
{
  std::list<std::string> lst;

  std::list<prefSection>::const_iterator it = data.begin();

  for ( ; it!=data.end(); it++)
  {
    if ( it->sectionName == section )
    {
      std::list<prefItem>::const_iterator it2 = it->items.begin();

      for ( ; it2!=it->items.end(); it2++)
      {
        lst.push_back( it2->name );
      }

      break;
    }
  }

  return lst;
}


