/*!
 * @file text.h
 * @brief Definition of a text Class, that is able to render text.
 */

#ifndef _TEXT_H
#define _TEXT_H

#include "element_2d.h"
#include "color.h"
#include "font.h"

#define  TEXT_ALIGN_LEFT             E2D_ALIGN_LEFT
#define  TEXT_ALIGN_RIGHT            E2D_ALIGN_RIGHT
#define  TEXT_ALIGN_CENTER           E2D_ALIGN_CENTER
#define  TEXT_ALIGN_SCREEN_CENTER    E2D_ALIGN_SCREEN_CENTER
#define  TEXT_DEFAULT_COLOR          Color(1.0, 1.0, 1.0, 1.0f) //!< the default Color (white, fully visible)

#define  TEXT_DEFAULT_ALIGNMENT      TEXT_ALIGN_LEFT            //!< default alignment
#define  TEXT_DEFAULT_SIZE           20                         //!< default size of the Text

// FORWARD DECLARATION
struct SDL_Surface;

//! Represents one textElement.
class Text : public Element2D
{
  ObjectListDeclaration(Text);
  public:
    Text(const std::string& fontFile = "", unsigned int fontSize = TEXT_DEFAULT_SIZE);
    Text(const Text& text);
    virtual ~Text();
    bool operator==(const Text& text) const;
    bool operator==(const std::string& text) const;
    Text& operator=(const Text& text);

    /// Final Interfacing.
    void setText(const std::string& text);
    void append(const std::string& appendText);
    void appendCharacter(char character);
    const std::string& operator<<(const std::string& appendText);
    void removeCharacters(unsigned int chars);
    void clear();

    /// SETUP
    void setFont(const std::string& fontFile, unsigned int renderSize);
    void setFont(const Font& font);
    /** @param blending the blending intensity to set (between 0.0 and 1.0) */
    inline void setBlending(float blending) { this->_font.setTransparency(blending); };
    /** @param r red @param g green @param b blue @brief sets the Color of the Text to render (values in [0-1]) */
    void setColor(float r, float g, float b) { this->_font.setDiffuseColor(Color(r, g, b, this->blending())); };
    void setColor(float r, float g, float b, float a) { this->_font.setDiffuseColor(Color(r, g, b, a)); };
    void setColor(const Color& color) { this->_font.setDiffuseColor(color); };
    void setSize(float size);


    /// RETRIEVE
    /** @returns the String this Text displays */
    inline const std::string& text() const { return this->_text; };

    /** @returns the pointer to the stored Font (not changeable) */
    inline const Font& font() const { return this->_font; };
    /** @returns the Blending Value [0 invisible 1.0 full visible */
    inline float blending() const { return this->_font.diffuseColor().a(); };
    /** @returns: a Vector(r,g,b) @brief: retrieve a Vector holding the Color of the Text */
    inline const Color& color() const { return this->_font.diffuseColor(); };
    /** @returns the Size of the Text */
    inline float size() const { return this->_size; };

    virtual void draw() const;

    void debug() const;

  protected:
    virtual void setupTextWidth();
  private:
    void init();

  private:
    Font              _font;           //!< Font of this text

    std::string       _text;           //!< The text to display
    float             _size;           //!< The size of the Text.
};

#endif /* _TEXT_H */
