/*!
 * @file font_data.h
 * @brief Contains the font-data class, that handles the reading of Images into Texutre-files.
 */

#ifndef _FONT_DATA_H
#define _FONT_DATA_H

#include "base_object.h"

#include "glincl.h"
#include "texture_data.h"

#ifdef HAVE_SDL_TTF_H
#include <SDL_ttf.h>
#else
#include <SDL/SDL_ttf.h>
#endif



/* some default values */
#define FONT_NUM_COLORS              256           //!< number of colors.

#define FONT_HIGHEST_KNOWN_CHAR      128           //!< The highest character known to the textEngine.
#define FONT_DEFAULT_RENDER_SIZE     50            //!< At what Resolution to render fonts.

class FontData
{


public:

//! A struct for handling glyphs
/**
 * a Glyph is one letter of a certain font
 */
  struct Glyph
  {
  // Glyph-specific (size and so on)
    Uint16   character;         //!< The character
    float    minX;              //!< The minimum distance from the origin in X
    float    maxX;              //!< The maximum distance from the origin in X
    float    minY;              //!< The minimum distance from the origin in Y
    float    maxY;              //!< The maximum distance from the origin in Y
    float    width;             //!< The width of the Glyph
    float    height;            //!< The height of the Glyph
    float    bearingX;          //!< How much is right of the Origin
    float    bearingY;          //!< How much is above the Origin
    float    advance;           //!< How big a Glyph would be in monospace-mode

    GLfloat texCoord[4];        //!< Texture coordinates: 0:left, 1:right, 2: top, 3: bottom.
  };

  typedef CountPointer<FontData> Pointer;
public:
  FontData();
  ~FontData();

  /// LOADING new Fonts
  bool loadFontFromTTF(const std::string& fontFile, unsigned int renderSize);
  bool loadFontFromSDL_Surface(SDL_Surface* surface);

  void setStyle(TTF_Font* font, const std::string& renderStyle);

  /** @returns a Pointer to the Array of Glyphs */
  inline const Glyph* const * const getGlyphArray() const { return this->glyphArray; };

  int getMaxHeight() const { return maxHeight; };
  int getMaxAscent() const { return maxAscent; };
  int getMaxDescent() const { return maxDescent; };

  /** @returns the Texture-Data of this FontData */
  const TextureData::Pointer& textureData() const { return texData; };

  bool rebuild() { return texData->rebuild(); };

private:
  void initGlyphs(TTF_Font* font, Uint16 from, Uint16 count);
  bool getGlyphMetrics(TTF_Font* font, Glyph* glyph, Uint16 character);

  int findOptimalFastTextureSize();
  bool createFastTexture(TTF_Font* font);


private:
  std::string   fontFile;            //!< The FileName the Font was loaded from.
  int           renderStyle;         //!< The Renderstyle
  unsigned int  renderSize;          //!< How big the Font should be rendered.

  Glyph**       glyphArray;          //!< An Array of all the Glyphs stored in the Array of Glyphs.

  int           maxHeight;           //!< Max Height of the Font.
  int           maxAscent;           //!< Max Ascent of the Font.
  int           maxDescent;          //!< Max Desent of the Font.

  TextureData::Pointer   texData;
};

#endif /* _FONT_DATA_H */
