/*!
 * @file font.h
 * brief Definition of the FONT-loading class
 *
 * !! IMPORTANT !! When using ttf fonts clear the license issues prior to
 * adding them to orxonox. This is really important, because we do not want
 * to offend anyone.
 */

#ifndef _FONT_H
#define _FONT_H

#include "material.h"

#include "font_data.h"


//! A class to handle a Font of a certain ttf-File/image-file, Size.
class Font : public Material
{
  ObjectListDeclaration(Font);
public:
  typedef FontData::Glyph Glyph;
public:
  Font();
  Font(const std::string& fontFile, unsigned int renderSize);
  Font(const std::string& imageFile);
  Font(char** xpmArray);
  Font(const Font& font);
  virtual ~Font();

  Font& operator=(const Font& font);
  /** @brief compare two fonts @param font the comparator, @returns true if they match */
  bool operator==(const Font& font) const { return this->data == font.data; };

  /// LOADING new Fonts
  bool loadFontFromTTF(const std::string& fontFile, unsigned int renderSize);
  bool loadFontFromSDL_Surface(SDL_Surface* surface);

  void setStyle(const std::string& renderStyle);

  /** @returns a Pointer to the Array of Glyphs */
  inline const Glyph* const * const getGlyphArray() const { return this->data->getGlyphArray(); };

  inline int getMaxHeight() const { return data->getMaxHeight(); };
  inline int getMaxAscent() const { return data->getMaxAscent(); };
  inline int getMaxDescent() const { return data->getMaxDescent(); };


  static void createAsciiImage(const std::string& ttfFile, const std::string& fileName, unsigned int size);

  void debug() const;

  void acquireData(const FontData::Pointer& data) { this->data = data; };
  const FontData::Pointer& dataPointer() const { return data; };
private:
  void init();
  static void initDefaultFont();

  void setTexture(const TextureData::Pointer& texDataPointer);

private:
  FontData::Pointer         data;                //!< A Data-Pointer to a Font.

  static FontData::Pointer  defaultFontData;     //!< a default font, that is used, if other fonts were unable to be loaded.
};

#endif /* _FONT_H */
