/*!
 * @file static_model.h
 * @brief Contains the Model Class that handles Static 3D-Models rendered with glList's
 */

#ifndef _STATIC_MODEL_H
#define _STATIC_MODEL_H

#include "model.h"

#include "material.h"
#include <vector>
#include "static_model_data.h"

/////////////
/// MODEL ///
/////////////
//! Class that handles static 3D-Models.
/**
 * it can also read them in and display them.
 * All the objects are rendered with glLists
 */
class StaticModel : public Model
{
  ObjectListDeclaration(StaticModel);
public:
  StaticModel(const std::string& modelName = "");
  StaticModel(const StaticModel& staticModel);
  virtual ~StaticModel();

  StaticModel& operator=(const StaticModel& model);

  virtual void draw() const { data->draw(); };
  void draw(int groupNumber) const { data->draw(groupNumber); };
  void draw(const std::string& groupName) const { data->draw(groupName); };

  void rebuild() { data->rebuild(); };

  Material* addMaterial(Material* material) { return data->addMaterial(material); };
  Material* addMaterial(const std::string& materialName) { return data->addMaterial(materialName); };

  bool addGroup(const std::string& groupString) { return data->addGroup(groupString); };

  bool addVertex(const std::string& vertexString) { return data->addVertex(vertexString); };
  bool addVertex(float x, float y, float z) { return data->addVertex(x, y, z); };

  bool addFace(const std::string& faceString) { return data->addFace(faceString); };
  bool addFace(int faceElemCount, VERTEX_FORMAT type, ...);

  bool addVertexNormal(const std::string& normalString) { return this->data->addVertexNormal(normalString); };
  bool addVertexNormal(float x, float y, float z) { return this->data->addVertexNormal(x,y,z); };

  bool addVertexTexture(const std::string& vTextureString) { return this->data->addVertexTexture(vTextureString); };
  bool addVertexTexture(float u, float v) { return this->data->addVertexTexture(u, v); };

  bool setMaterial(const std::string& mtlString) { return data->setMaterial(mtlString); };
  bool setMaterial(Material* mtl) { return data->setMaterial(mtl);};

  void finalize();

  void acquireData(const StaticModelData::Pointer& data);
  const StaticModelData::Pointer& dataPointer() const { return this->data; };

  inline void setScaleFactor(float scaleFactor) { this->data->setScaleFactor(scaleFactor); };
  float getScaleFactor() const { return data->getScaleFactor(); }

protected:
  void cubeModel();

private:
  void updateBase();
private:
  StaticModelData::Pointer         data;
};

#endif
