/*!
 * @file resource.h
 * @brief Definition of a NewResource.
*/

#ifndef _RESOURCE_H
#define _RESOURCE_H

#include "base_object.h"
#include <string>
#include <vector>
#include <set>

#include "filesys/directory.h"

//! A NewResource is an Object, that can be loaded from Disk
/**
 *
 */
class NewResource : virtual public BaseObject
{
  ObjectListDeclaration(NewResource);
public:
  class KeepLevel
  {
  public:
    KeepLevel(unsigned int keepLevel) { _keepLevel = keepLevel; };
    KeepLevel(const std::string& keepLevelName);

    static void defineKeepLevelName(unsigned int level, const std::string& name);
  private:
    unsigned int                        _keepLevel;
  private:
    static std::vector<std::string>     _keepLevelNames;
  };


protected:
  class StorePointer
  {
  public:
    StorePointer(const std::string& loadString, const NewResource::KeepLevel& keeplevel);
    const std::string& loadString() const { return _loadString; };
    const NewResource::KeepLevel& keepLevel() const { return _keepLevel; };

  private:
    std::string                 _loadString;             //!< An identifier, to match when loading a File.
    NewResource::KeepLevel      _keepLevel;              //!< The Priority of this resource. (can only be increased, so none else will delete this)
  };


  class Type
  {
  public:
    Type(const ClassID& classID) : _id(-1), _classID(classID) { };

    void addExtension(const std::string& extension);

    void addResourcePath(const std::string& path);
    void addResourceSubPath(const std::string& subPath);

    /// Retrieve Functions
    const ClassID& storedClassID() const { return _classID; };
    int id() const { return _id; };
    const std::vector<std::string>& resourcePaths() const { return _resourcePaths; };
    const std::vector<std::string>& resourceSubPaths() const { return _resourceSubPaths; };
    const std::vector<NewResource::StorePointer*>& storedResources() const { return _storedResources; };

    void setID(int id);
    void addResource(NewResource::StorePointer* resource);

  private:
    int                             _id;
    const ClassID&                  _classID;
    std::vector<std::string>        _resourcePaths;
    std::vector<std::string>        _resourceSubPaths;
    std::vector<std::string>        _fileExtensions;

    std::vector<NewResource::StorePointer*> _storedResources;
  };


public:
  NewResource(NewResource::Type* type);
  virtual ~NewResource();

  virtual bool reload() { return false; };
  virtual bool unload() { return false; };

  std::string locateFile(const std::string& fileName);


public:
  static void setMainGlobalPath(const Directory& directory);
  static void addGlobalPath(const Directory& directory);


protected:
  NewResource::StorePointer* acquireResource(const std::string& loadString);
  void addResource(NewResource::StorePointer* pointer);

private:
  NewResource::StorePointer*       _pointer;                         //!< Virtual Pointer to the ResourceData.
  NewResource::Type*          _type;                            //!< Type of the NewResource.


  static std::vector<NewResource::Type*>    _resourceTypes;

  //! GLOBALS
  static Directory                       _mainGlobalPath;
  static std::vector<Directory>          _globalPaths;
};

#endif /* _RESOURCE_H */
