/*!
 * @file executor_substring.h
 * Definition of a Generic Executor
 */

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/



#ifndef __EXECUTOR_SUBSTRING_H_
#define __EXECUTOR_SUBSTRING_H_


#include "executor_generic.h"
#include "substring.h"
#ifdef FUNCTOR_CALL_TYPE
 #undef FUNCTOR_CALL_TYPE
#endif
//! Use SubString as the FUNCTION_CALL_TYPE so that we can easily extend the Engine.
#define FUNCTOR_CALL_TYPE const SubString

/**
 * @brief converts an input string into a value, if no value was converted, uses defaultValue.
 * @param input the input as a String.
 * @param defaultValue the Default Value set, if the input could not be converted into type.
 * @returns either the converted value or defaultValue.
 */
template<typename type> type fromString(const std::string& input, type defaultValue) { return defaultValue; };
template<> bool fromString<bool>(const std::string& input, bool defaultValue);
template<> int fromString<int>(const std::string& input, int defaultValue);
template<> unsigned int fromString<unsigned int>(const std::string& input, unsigned int defaultValue);
template<> float fromString<float>(const std::string& input, float defaultValue);
template<> char fromString<char>(const std::string& input, char defaultValue);
template<> const std::string& fromString<const std::string&>(const std::string& input, const std::string& defaultValue);

/**
 * @brief converts to any type from a MultiType
 * @param multi the MultiType to convert.
 * @returns the converted Value.
 */
template<typename type> type fromMulti(const MultiType& multi) { /* return defaultValue; */ };
template<> bool fromMulti<bool>(const MultiType& multi);
template<> int fromMulti<int>(const MultiType& multi);
template<> unsigned int fromMulti<unsigned int>(const MultiType& multi);
template<> float fromMulti<float>(const MultiType& multi);
template<> char fromMulti<char>(const MultiType& multi);
template<> const std::string& fromMulti<const std::string&>(const MultiType& multi);

/**
 * @brief retrieves a default value from an array of default values, at possition i.
 * @param defaultValues the Array of default values.
 * @param i the index.
 * @returns the Default Value at Position i
 */
template<typename type> type getDefault(const MultiType* const defaultValues, unsigned int i) { return (type)0; };
template<> bool getDefault<bool>(const MultiType* const defaultValues, unsigned int i);
template<> int getDefault<int>(const MultiType* const defaultValues, unsigned int i);
template<> unsigned int getDefault<unsigned int>(const MultiType* const defaultValues, unsigned int i);
template<> float getDefault<float>(const MultiType* const defaultValues, unsigned int i);
template<> char getDefault<char>(const MultiType* const defaultValues, unsigned int i);
template<> const std::string& getDefault<const std::string&>(const MultiType* const defaultValues, unsigned int i);


/**
 * @brief to remove writing errors, this function is Used.
 * @param sub The SubString to use
 * @param default The default Values.
 */
template<> class ExecutorEvaluater <const SubString>
{
public:
  /** @brief Executes the Evaluater
   * @param CallValue the Value that should be converted
   * @param defaults the default Values.
   */
  template <typename ToType, int index>
  static ToType getValue(const SubString& CallValue, const MultiType* const defaults)
  {
    return (CallValue.size() > index) ?
           fromString<ToType>(CallValue[index], getDefault<ToType>(defaults, index)) :
           fromMulti<ToType>(defaults[index]);
  }
  /** @returns the default Value of a SubString, namely NullSubString or SubString() */
  static const SubString& defaultValue() { return SubString::NullSubString; };
};

#endif /* __EXECUTOR_SUBSTRING_H_ */
