/*!
 * @file executor_lua_state.h
 * Definition of a Executor that takes lua_State* as input.
 */

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/



#ifndef __EXECUTOR_LUA_STATE_H_
#define __EXECUTOR_LUA_STATE_H_


#include "executor_generic.h"

#include "luaincl.h"

#ifdef FUNCTOR_CALL_TYPE
 #undef FUNCTOR_CALL_TYPE
#endif
//! Define the Functor call type as lua_State*.
#define FUNCTOR_CALL_TYPE lua_State*

template<typename type> type fromLua(lua_State* state, int index);
template<> bool fromLua<bool>(lua_State* state, int index);
template<> int fromLua<int>(lua_State* state, int index);
template<> unsigned int fromLua<unsigned int>(lua_State* state, int index);
template<> float fromLua<float>(lua_State* state, int index);
template<> char fromLua<char>(lua_State* state, int index);
template<> const std::string& fromLua<const std::string&>(lua_State* state, int index);

template<typename type> void toLua(lua_State* state, type value);
template<> void toLua<bool>(lua_State* state, bool value);
template<> void toLua<int>(lua_State* state, int value);
template<> void toLua<unsigned int>(lua_State* state, unsigned int value);
template<> void toLua<float>(lua_State* state, float value);
template<> void toLua<char>(lua_State* state, char value);
template<> void toLua<const std::string&>(lua_State* state, const std::string& value);

//! A Class, that evaluates a lua_State and converts indices into different Types.
template<> class ExecutorEvaluater <lua_State*>
{
public:
  /** @brief Executes the Evaluater
   * @param CallValue the Value that should be converted
   * @param defaults the default Values.
   */
  template <typename ToType, int index>
  static ToType getValue(lua_State*& CallValue, const MultiType* const defaults)
  {
    return (fromLua<ToType>(CallValue, index+1));
  }
  /**
   * @param state the state to write into
   * @param value the Value to write there.
   */
  template <typename FromType>
  static void storeRet(lua_State*& state, FromType value)
  {
    toLua<FromType>(state, value);
  }
  /** @returns the Null Value of a lua_State*, namely (pointer-type) NULL */
  static lua_State*& defaultValue() { static lua_State* nullState; return nullState; };
};

#endif /* __EXECUTOR_LUA_STATE_H_ */
