/*!
 * @file executor_generic.h
 * Definition of a Generic Executor
 */

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/



#ifndef __EXECUTOR_GENERIC_H_
#define __EXECUTOR_GENERIC_H_


#include "executor.h"
/**
 * @brief this is a Template Class used as an evaluater.
 *
 * Trait to determine a default Value for any Type,
 * and to define the Convertible, and how it is transformed into the
 * corresponding SubTypes over the operator().
 *
 * This Class must be reimplemented for each Convertible and all of its
 *  conversion-members.
 *
 * e.g: Convertible SubSting, that splits up into many Stings
 *      conversion-members: (int) can be transformed from a String.
 */
template<typename FromType> class ExecutorEvaluater
{
public:
  /** @brief Executes the Evaluater
   * @param CallValue the Value that should be converted
   * @param defaults the default Values.
   */
  template <typename ToType, int index>
  static ToType getValue(FromType& CallValue, const MultiType* const defaults)
  {
    assert(0 && "Might want to extend the ExecutorEvaluater::getValue function in an 'Explicit Specialized Template Class'");
    return defaultValue;
  }
  /** @param @storeTo Stores the returned value here. @param value The Value to Store. @note invers of getValue */
  static void storeRet(FromType& storeTo, FromType value)
  {
    assert (0 && "Might want to extend the ExecutorEvaluater::storeRet function in an 'Explicit Specialized Template Class'");
  }
  /** @returns the Default Value of ToType */
  static FromType& defaultValue() { static FromType defaultValue; return defaultValue; };
};
#endif /* __EXECUTOR_GENERIC_H_ */


#ifdef __EXECUTOR_FUNCTIONAL_NAME
///////////
//// 0 ////
///////////
//! @brief ExecutorClass, that can execute Functions without any parameters.
template<class T, typename CallType, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(0,) :public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(0,) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST )
      : Executor<CallType, BaseClass>(false, __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE), functionPointer(functionPointer)
  {};

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param eval the CallType to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& eval = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)();
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(0,)<T, CallType>(this->functionPointer);
  };
};



///////////
//// 1 ////
///////////
//! @brief ExecutorClass, that can execute Functions with one parameter.
template<class T, typename CallType, typename type0, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(1,) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(1,) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE), functionPointer(functionPointer)
  {};

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param eval the CallType to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& eval = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
	Evaluater<CallType>::template getValue<type0, 0>(eval, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return  new __EXECUTOR_FUNCTIONAL_NAME(1,)<T, CallType, type0>(this->functionPointer);
  };
};



///////////
//// 2 ////
///////////
//! @brief ExecutorClass, that can execute Functions with two parameters.
template<class T, typename CallType, typename type0, typename type1, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(2,) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(2,) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE), functionPointer(functionPointer)
  {};

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param eval the CallType to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& eval = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType>::template getValue<type0, 0>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type1, 1>(eval, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(2,)<T, CallType, type0, type1>(this->functionPointer);
  };
};



///////////
//// 3 ////
///////////
//! @brief ExecutorClass, that can execute Functions with three parameters.
template<class T, typename CallType, typename type0, typename type1, typename type2, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(3,) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(3,) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE), functionPointer(functionPointer)
  {};

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param eval the CallType to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& eval = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType>::template getValue<type0, 0>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type1, 1>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type2, 2>(eval, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(3,)<T, CallType, type0, type1, type2>(this->functionPointer);
  };
};



///////////
//// 4 ////
///////////
//! @brief ExecutorClass, that can execute Functions with four parameters.
template<class T, typename CallType, typename type0, typename type1, typename type2, typename type3, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(4,) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(4,) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), ExecutorParamType<type3>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE), functionPointer(functionPointer)
  {};

  /**
  * @brief executes the Functional
  * @param object the Object the action should be executed on.
  * @param eval the CallType to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& eval = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType>::template getValue<type0, 0>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type1, 1>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type2, 2>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type3, 3>(eval, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(4,)<T, CallType, type0, type1, type2, type3>(this->functionPointer);
  };
};



///////////
//// 5 ////
///////////
//! @brief ExecutorClass, that can execute Functions with five parameters.
template<class T, typename CallType, typename type0, typename type1, typename type2, typename type3, typename type4, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(5,) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3, type4) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(5,) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3, type4) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), ExecutorParamType<type3>(), ExecutorParamType<type4>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE), functionPointer(functionPointer)
  {};

  /**
  * @brief executes the Functional
  * @param object the Object the action should be executed on.
  * @param eval the CallType to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& eval = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType>::template getValue<type0, 0>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type1, 1>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type2, 2>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type3, 3>(eval, this->defaultValue),
      Evaluater<CallType>::template getValue<type4, 4>(eval, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(5,)<T, CallType, type0, type1, type2, type3, type4>(this->functionPointer);
  };
};


////////////////////
//// 0 & RETURN ////
////////////////////
//! @brief ExecutorClass, that can execute Functions with one parameter.
template<class T, typename CallType, typename ReturnType, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(0,ret) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  ReturnType (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(0,ret) (ReturnType (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(true, __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE), functionPointer(functionPointer)
  {};

  /**
     * @brief executes the Functional
     * @param object the Object the action should be executed on.
     * @param eval the CallType to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& eval = Evaluater<CallType>::defaultValue()) const
  {
    Evaluater<CallType>::template storeRet<ReturnType>(eval, (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)());
  };

  /**
     * @brief copies the Executor
     * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return  new __EXECUTOR_FUNCTIONAL_NAME(0,ret)<T, CallType, ReturnType>(this->functionPointer);
  };
};

////////////////////
//// 1 & RETURN ////
////////////////////
//! @brief ExecutorClass, that can execute Functions with one parameter.
template<class T, typename CallType, typename ReturnType, typename type0, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(1,ret) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  ReturnType (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(1,ret) (ReturnType (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(true, ExecutorParamType<type0>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE), functionPointer(functionPointer)
  {};

  /**
     * @brief executes the Functional
     * @param object the Object the action should be executed on.
     * @param eval the CallType to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& eval = Evaluater<CallType>::defaultValue()) const
  {
    Evaluater<CallType>::template storeRet<ReturnType>(eval, (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
          Evaluater<CallType>::template getValue<type0, 0>(eval, this->defaultValue)));
  };

  /**
     * @brief copies the Executor
     * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return  new __EXECUTOR_FUNCTIONAL_NAME(1,ret)<T, CallType, ReturnType, type0>(this->functionPointer);
  };
};


////////////////////
//// 2 & RETURN ////
////////////////////
//! @brief ExecutorClass, that can execute Functions with one parameter.
template<class T, typename CallType, typename ReturnType, typename type0, typename type1, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(2,ret) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  ReturnType (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(2,ret) (ReturnType (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(true, ExecutorParamType<type0>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE), functionPointer(functionPointer)
  {};

  /**
     * @brief executes the Functional
     * @param object the Object the action should be executed on.
     * @param eval the CallType to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& eval = Evaluater<CallType>::defaultValue()) const
  {
    Evaluater<CallType>::template storeRet<ReturnType>(eval, (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
          Evaluater<CallType>::template getValue<type0, 0>(eval, this->defaultValue),
          Evaluater<CallType>::template getValue<type1, 1>(eval, this->defaultValue)));
  };

  /**
     * @brief copies the Executor
     * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return  new __EXECUTOR_FUNCTIONAL_NAME(2,ret)<T, CallType, ReturnType, type0, type1>(this->functionPointer);
  };
};

#endif /* __EXECUTOR_FUNCTIONAL_NAME */
