/*!
 * @file executor_generic.h
 * Definition of a Generic Executor
 */

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/


#ifndef __EXECUTOR_GENERIC_H_
#define __EXECUTOR_GENERIC_H_

#include "executor.h"
#include "substring.h"


template<typename type> type fromString(const std::string& input, type defaultValue) { return defaultValue; };
template<> bool fromString<bool>(const std::string& input, bool defaultValue);
template<> int fromString<int>(const std::string& input, int defaultValue);
template<> unsigned int fromString<unsigned int>(const std::string& input, unsigned int defaultValue);
template<> float fromString<float>(const std::string& input, float defaultValue);
template<> char fromString<char>(const std::string& input, char defaultValue);
template<> const std::string& fromString<const std::string&>(const std::string& input, const std::string& defaultValue);

template<typename type> type fromMulti(const MultiType& multi) { /* return defaultValue; */ };
template<> bool fromMulti<bool>(const MultiType& multi);
template<> int fromMulti<int>(const MultiType& multi);
template<> unsigned int fromMulti<unsigned int>(const MultiType& multi);
template<> float fromMulti<float>(const MultiType& multi);
template<> char fromMulti<char>(const MultiType& multi);
template<> const std::string& fromMulti<const std::string&>(const MultiType& multi);


template<typename type> type getDefault(const MultiType* const defaultValues, unsigned int i) { return (type)0; };
template<> bool getDefault<bool>(const MultiType* const defaultValues, unsigned int i);
template<> int getDefault<int>(const MultiType* const defaultValues, unsigned int i);
template<> unsigned int getDefault<unsigned int>(const MultiType* const defaultValues, unsigned int i);
template<> float getDefault<float>(const MultiType* const defaultValues, unsigned int i);
template<> char getDefault<char>(const MultiType* const defaultValues, unsigned int i);
template<> const std::string& getDefault<const std::string&>(const MultiType* const defaultValues, unsigned int i);


/**
 * @brief this is a Template Class used as an evaluater.
 *
 * Trait to determine a default Value for any Type,
 * and to define the Convertible, and how it is transformed into the
 * corresponding SubTypes over the operator().
 *
 * This Class must be reimplemented for each Convertible and all of its
 *  conversion-members.
 *
 * e.g: Convertible SubSting, that splits up into many Stings
 *      conversion-members: (int) can be transformed from a String.
 */
template<typename FromType> class ExecutorEvaluater
{
public:
  /** @brief Executes the Evaluater
   * @param CallValue the Value that should be converted
   * @param defaults the default Values.
   */
  template <typename ToType, int index>
  ToType operator()(FromType& CallValue, const MultiType* const defaults)
  {
    return defaultValue; /*(CallValue.size() > index) ?
           fromString<ToType>(CallValue[index], getDefault<ToType>(defaults, index)) :
    fromMulti<ToType>(defaults[index]); */
  }
  static FromType& defaultValue() { return FromType(); };
};

/**
 * @brief to remove writing errors, this function is Used.
 * @param sub The SubString to use
 * @param default The default Values.
 */
template<> class ExecutorEvaluater <const SubString>
{
public:
  /** @brief Executes the Evaluater
   * @param CallValue the Value that should be converted
   * @param defaults the default Values.
   */
  template <typename ToType, int index>
  ToType operator()(const SubString& CallValue, const MultiType* const defaults)
  {
    return (CallValue.size() > index) ?
           fromString<ToType>(CallValue[index], getDefault<ToType>(defaults, index)) :
           fromMulti<ToType>(defaults[index]);
  }
  static const SubString& defaultValue() { return SubString::NullSubString; };
};


///////////
//// 0 ////
///////////
//! @brief ExecutorClass, that can execute Functions without any parameters.
template<class T, typename CallType, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(0) :public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(0) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST )
      : Executor<CallType, BaseClass>(false, __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE)
  {
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)();
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(0)<T, CallType>(this->functionPointer);
  };
};





///////////
//// 1 ////
///////////
//! @brief ExecutorClass, that can execute Functions with one parameter.
template<class T, typename CallType, typename type0, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(1) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(1) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE)
  {
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType>().template operator()<type0, 0>(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return  new __EXECUTOR_FUNCTIONAL_NAME(1)<T, CallType, type0>(this->functionPointer);
  };
};





///////////
//// 2 ////
///////////
//! @brief ExecutorClass, that can execute Functions with two parameters.
template<class T, typename CallType, typename type0, typename type1, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(2) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(2) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE)
  {
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType>().template operator()<type0, 0>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type1, 1>(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(2)<T, CallType, type0, type1>(this->functionPointer);
  };
};





///////////
//// 3 ////
///////////
//! @brief ExecutorClass, that can execute Functions with three parameters.
template<class T, typename CallType, typename type0, typename type1, typename type2, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(3) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(3) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE)
  {
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType>().template operator()<type0, 0>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type1, 1>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type2, 2>(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(3)<T, CallType, type0, type1, type2>(this->functionPointer);
  };
};





///////////
//// 4 ////
///////////
//! @brief ExecutorClass, that can execute Functions with four parameters.
template<class T, typename CallType, typename type0, typename type1, typename type2, typename type3, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(4) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(4) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), ExecutorParamType<type3>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE)
  {
    this->functionPointer = functionPointer;
  };

  /**
  * @brief executes the Functional
  * @param object the Object the action should be executed on.
  * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType>().template operator()<type0, 0>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type1, 1>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type2, 2>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type3, 3>(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(4)<T, CallType, type0, type1, type2, type3>(this->functionPointer);
  };
};






///////////
//// 5 ////
///////////
//! @brief ExecutorClass, that can execute Functions with five parameters.
template<class T, typename CallType, typename type0, typename type1, typename type2, typename type3, typename type4, template<typename> class Evaluater = ExecutorEvaluater, class BaseClass = BaseObject>
class __EXECUTOR_FUNCTIONAL_NAME(5) : public Executor<CallType, BaseClass>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3, type4) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(5) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3, type4) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType, BaseClass>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), ExecutorParamType<type3>(), ExecutorParamType<type4>(), __EXECUTOR_FUNCTIONAL_FUNCTIONTYPE)
  {
    this->functionPointer = functionPointer;
  };

  /**
  * @brief executes the Functional
  * @param object the Object the action should be executed on.
  * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = Evaluater<CallType>::defaultValue()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType>().template operator()<type0, 0>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type1, 1>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type2, 2>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type3, 3>(sub, this->defaultValue),
      Evaluater<CallType>().template operator()<type4, 4>(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType, BaseClass>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(5)<T, CallType, type0, type1, type2, type3, type4>(this->functionPointer);
  };
};

#ifndef __EXECUTOR_GENERIC_H_
