/*!
 * @file executor.h
 * Definition of an Executor
 */

#ifndef _EXECUTOR_H
#define _EXECUTOR_H

#include "base_object.h"

#include "multi_type.h"

//! The maximum Count of Arguments of the Executor
/** This is Hardcoded, for each Executor. */
#define EXECUTOR_MAX_ARGUMENTS                7



template<typename type> MT_Type ExecutorParamType() { return MT_EXT1; };
template<> MT_Type ExecutorParamType<bool>();
template<> MT_Type ExecutorParamType<int>();
template<> MT_Type ExecutorParamType<unsigned int>();
template<> MT_Type ExecutorParamType<float>();
template<> MT_Type ExecutorParamType<char>();
template<> MT_Type ExecutorParamType<const std::string&>();

////////////////
// BASE CLASS //
////////////////
//! a BaseClass for all possible Executors
/**
 * An Executor is an Object, that is able to call Objects of Any type (class)
 * and execute a function with given parameters on it.
 *
 * The Executor is able to handle:
 *  Objects of any Class (Templated)
 *  Default Values
 *  Functions with up to 5 parameters (more seems overhead, split up the function)
 *  Functions with many types (@see functor_list.h)
 */
template <typename CallType, class BaseClass = BaseObject> class Executor
{
public:
  //! an enumerator for the definition of the Type.
  typedef enum {
    FunctionMember,      //!< The function is neither Static nor Constant
    FunctionStatic,      //!< The Function is Static and pointing to either a Static Member or a C-style function.
    FunctionConstMember, //!< The Function is Constant and pointing to a Member that does not change the Object.
  } FunctionType;

public:
  virtual ~Executor() {};

  // RETRIEVE INFORMATION
  /** @param i the i'th defaultValue, @returns reference to the MultiType */
  inline MultiType& getDefaultValue(unsigned int i) { return defaultValue[i]; };
  /** @returns the default Values as a List */
  inline const MultiType* const getDefaultValues() { return defaultValue; };

  /** @returns the Type of this Function (either static or objective) */
  inline FunctionType getType() const { return this->functionType; };

  /** @returns the Count of Parameters this Executor takes */
  inline unsigned int getParamCount() const { return this->paramCount; };
  /** @returns true if the Executor has a return Value. */
  inline bool hasRetVal() const { return bRetVal; };

  /** executes a Command. @param objec the Object, @param count how many values, @param values the Values */
  virtual void operator()(BaseClass* object, CallType& values) const = 0;

  /**
   * @brief set the default values of the executor
   * @param value0 the first default value
   * @param value1 the second default value
   * @param value2 the third default value
   * @param value3 the fourth default value
   * @param value4 the fifth default value
   * @returns itself
   * @note: THIS FUNCTION WILL BE REPLACED BY A CONFIGURATOR (most probably).
  */
  Executor* defaultValues(const MultiType& value0 = MT_NULL, const MultiType& value1 = MT_NULL,
                          const MultiType& value2 = MT_NULL, const MultiType& value3 = MT_NULL,
                          const MultiType& value4 = MT_NULL, const MultiType& value5 = MT_NULL,
                          const MultiType& value6 = MT_NULL)
  {
    const MultiType* value[5];
    value[0] = &value0;
    value[1] = &value1;
    value[2] = &value2;
    value[3] = &value3;
    value[4] = &value4;
    value[5] = &value5;
    value[6] = &value6;
    for (unsigned int i = 0; i < this->paramCount; i++)
    {
      if (*value[i] != MT_NULL)
      {
        this->defaultValue[i].setValueOf(*value[i]);
        this->defaultValue[i].storeString();
      }
    }
    return this;
  }

  /** @returns the Clone as a new Copy of the Executor. */
  virtual Executor<CallType>* clone () const = 0;


protected:
  //! Now follows a List of Executor Constructors, to be fast in creating.
  Executor(bool hasRetVal, FunctionType functionType = FunctionMember)
  : bRetVal(hasRetVal), paramCount(0), functionType(functionType)
  { };

  Executor(bool hasRetVal, const MultiType& param0,
	   FunctionType functionType = FunctionMember)
  : bRetVal(hasRetVal), paramCount(1), functionType(functionType)
  {
    this->defaultValue[0] = param0;
  };

  Executor(bool hasRetVal, const MultiType& param0, const MultiType& param1,
	   FunctionType functionType = FunctionMember)
  : bRetVal(hasRetVal), paramCount(2), functionType(functionType)
  {
    this->defaultValue[0] = param0;
    this->defaultValue[1] = param1;
  };

  Executor(bool hasRetVal, const MultiType& param0, const MultiType& param1,
	   const MultiType& param2,
	   FunctionType functionType = FunctionMember)
  : bRetVal(hasRetVal), paramCount(3), functionType(functionType)
  {
    this->defaultValue[0] = param0;
    this->defaultValue[1] = param1;
    this->defaultValue[2] = param2;
  };

  Executor(bool hasRetVal, const MultiType& param0, const MultiType& param1,
	   const MultiType& param2, const MultiType& param3,
	   FunctionType functionType = FunctionMember)
  : bRetVal(hasRetVal), paramCount(4), functionType(functionType)
  {
    this->defaultValue[0] = param0;
    this->defaultValue[1] = param1;
    this->defaultValue[2] = param2;
    this->defaultValue[3] = param3;
  };

  Executor(bool hasRetVal, const MultiType& param0, const MultiType& param1,
	   const MultiType& param2, const MultiType& param3,
	   const MultiType& param4,
	   FunctionType functionType = FunctionMember)
  : bRetVal(hasRetVal), paramCount(5), functionType(functionType)
  {
    this->defaultValue[0] = param0;
    this->defaultValue[1] = param1;
    this->defaultValue[2] = param2;
    this->defaultValue[3] = param3;
    this->defaultValue[4] = param4;
  };

  Executor(bool hasRetVal, const MultiType& param0, const MultiType& param1,
	   const MultiType& param2, const MultiType& param3,
	   const MultiType& param4, const MultiType& param5,
	   FunctionType functionType = FunctionMember)
  : bRetVal(hasRetVal), paramCount(6), functionType(functionType)
  {
    this->defaultValue[0] = param0;
    this->defaultValue[1] = param1;
    this->defaultValue[2] = param2;
    this->defaultValue[3] = param3;
    this->defaultValue[4] = param4;
    this->defaultValue[5] = param5;
  };

  Executor(bool hasRetVal, const MultiType& param0, const MultiType& param1,
	   const MultiType& param2, const MultiType& param3,
	   const MultiType& param4, const MultiType& param5,
	   const MultiType& param6,
	   FunctionType functionType = FunctionMember)
  : bRetVal(hasRetVal), paramCount(7), functionType(functionType)
  {
    this->defaultValue[0] = param0;
    this->defaultValue[1] = param1;
    this->defaultValue[2] = param2;
    this->defaultValue[3] = param3;
    this->defaultValue[4] = param4;
    this->defaultValue[5] = param5;
    this->defaultValue[6] = param6;
  };

protected:
  const bool                  bRetVal;          //!< True if the Executor has a return Value.
  const unsigned int          paramCount;       //!< the count of parameters.
  MultiType                   defaultValue[7];  //!< Default Values.

  const FunctionType          functionType;     //!< What Type of Function it is.
};

#endif /* _EXECUTOR_H */
