/*!
 * @file executor_functional.h
 * Definition of an Executor
 */

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/


#ifndef __EXECUTOR_FUNCTIONAL_H_
#define __EXECUTOR_FUNCTIONAL_H_

#include "executor.h"
#include "substring.h"


template<typename type> type fromString(const std::string& input, type defaultValue) { return defaultValue; };
template<> bool fromString<bool>(const std::string& input, bool defaultValue);
template<> int fromString<int>(const std::string& input, int defaultValue);
template<> unsigned int fromString<unsigned int>(const std::string& input, unsigned int defaultValue);
template<> float fromString<float>(const std::string& input, float defaultValue);
template<> char fromString<char>(const std::string& input, char defaultValue);
template<> const std::string& fromString<const std::string&>(const std::string& input, const std::string& defaultValue);

template<typename type> type fromMulti(const MultiType& multi) { /* return defaultValue; */ };
template<> bool fromMulti<bool>(const MultiType& multi);
template<> int fromMulti<int>(const MultiType& multi);
template<> unsigned int fromMulti<unsigned int>(const MultiType& multi);
template<> float fromMulti<float>(const MultiType& multi);
template<> char fromMulti<char>(const MultiType& multi);
template<> const std::string& fromMulti<const std::string&>(const MultiType& multi);


template<typename type> type getDefault(const MultiType* const defaultValues, unsigned int i) { return (type)0; };
template<> bool getDefault<bool>(const MultiType* const defaultValues, unsigned int i);
template<> int getDefault<int>(const MultiType* const defaultValues, unsigned int i);
template<> unsigned int getDefault<unsigned int>(const MultiType* const defaultValues, unsigned int i);
template<> float getDefault<float>(const MultiType* const defaultValues, unsigned int i);
template<> char getDefault<char>(const MultiType* const defaultValues, unsigned int i);
template<> const std::string& getDefault<const std::string&>(const MultiType* const defaultValues, unsigned int i);


/**
 * @brief to remove writing errors, this function is Used.
 * @param sub The SubString to use
 * @param default The default Values.
 */
template<typename FromType, typename ToType, int index> class ExecutorEvaluater
{
public:
  ToType operator()(FromType& CallValue, const MultiType* const defaults)
  {
    return (CallValue.size() > index) ?
           fromString<ToType>(CallValue[index], getDefault<ToType>(defaults, index)) :
           fromMulti<ToType>(defaults[index]);
  }
};

#endif /* __EXECUTOR_FUNCTIONAL_H_ */

//! if Functional is constant calling
#define __EXECUTOR_FUNCTIONAL_CONST
//! The Name to be attached to the functional (for normal, static, and const modes)
#define __EXECUTOR_FUNCTIONAL_NAME(ParamCount)   Executor##ParamCount##Params
//! The Execution-mode (either static or objective)
#define __EXECUTOR_FUNCTIONAL_FUNCTION_EXEC      dynamic_cast<T*>(object)->*(functionPointer)
//! The Function-Pointer, and how to save it internally.
#define __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER   T::*functionPointer

#ifdef EXECUTOR_FUNCTIONAL_USE_CONST            //! USING CONST FUNCTIONALS
 #ifdef __EXECUTOR_FUNCTIONAL_CONST_INCLUDED    //!< CHECK IF ALREADY INCLUED CONST FUNCTIONALS
  #define __EXECUTOR_FUNCTIONAL_NOT_INCLUDE_THIS
 #else
  #define __EXECUTOR_FUNCTIONAL_CONST_INCLUDED
 #endif
 #undef __EXECUTOR_FUNCTIONAL_CONST
 #define __EXECUTOR_FUNCTIONAL_CONST const
 #undef __EXECUTOR_FUNCTIONAL_NAME
 #define __EXECUTOR_FUNCTIONAL_NAME(ParamCount) Executor##ParamCount##Params_const
//#endif /* EXECUTOR_FUNCTIONAL_USE_CONST */

#elif defined EXECUTOR_FUNCTIONAL_USE_STATIC
 #ifdef EXECUTOR_FUNCTIONAL_USE_CONST
  #error you obviously do not know what you are doing !! ask the bensch
 #endif

#ifdef __EXECUTOR_FUNCTIONAL_STATIC_INCLUDED
   #define __EXECUTOR_FUNCTIONAL_NOT_INCLUDE_THIS
  #else
   #define __EXECUTOR_FUNCTIONAL_STATIC_INCLUDED
  #endif

#undef __EXECUTOR_FUNCTIONAL_FUNCTION_EXEC
 #define __EXECUTOR_FUNCTIONAL_FUNCTION_EXEC       functionPointer
 #undef __EXECUTOR_FUNCTIONAL_NAME
 #define __EXECUTOR_FUNCTIONAL_NAME(ParamCount)    Executor##ParamCount##Params_static
 #undef __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER
 #define __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER    *functionPointer

//#endif /* EXECUTOR_FUNCTIONAL_USE_STATIC */
#else
 #ifdef __EXECUTOR_FUNCTIONAL_PLAIN_INCLUDED
  #define __EXECUTOR_FUNCTIONAL_NOT_INCLUDE_THIS
 #else
  #define __EXECUTOR_FUNCTIONAL_PLAIN_INCLUDED
 #endif
#endif

#ifndef __EXECUTOR_FUNCTIONAL_NOT_INCLUDE_THIS

///////////
//// 0 ////
///////////
//! @brief ExecutorClass, that can execute Functions without any parameters.
template<class T, typename CallType, template<typename, typename, int> class Evaluater = ExecutorEvaluater>
class __EXECUTOR_FUNCTIONAL_NAME(0) :public Executor<CallType>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(0) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST )
      : Executor<CallType>(false)
  {
    this->functionType = Executor<CallType>::FunctionDefault;
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)();
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(0)<T, CallType>(this->functionPointer);
  };
};



///////////
//// 1 ////
///////////
//! @brief ExecutorClass, that can execute Functions with one parameter.
template<class T, typename CallType, typename type0, template<typename, typename, int> class Evaluater = ExecutorEvaluater>
class __EXECUTOR_FUNCTIONAL_NAME(1) : public Executor<CallType>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(1) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType>(false, ExecutorParamType<type0>())
  {
    this->functionType = Executor<CallType>::FunctionDefault;
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType, type0, 0>()(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType>* clone() const
  {
    return  new __EXECUTOR_FUNCTIONAL_NAME(1)<T, CallType, type0>(this->functionPointer);
  };
};

///////////
//// 2 ////
///////////
//! @brief ExecutorClass, that can execute Functions with two parameters.
template<class T, typename CallType, typename type0, typename type1, template<typename, typename, int> class Evaluater = ExecutorEvaluater>
class __EXECUTOR_FUNCTIONAL_NAME(2) : public Executor<CallType>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(2) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>())
  {
    this->functionType = Executor<CallType>::FunctionDefault;
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType, type0, 0>()(sub, this->defaultValue),
      Evaluater<CallType, type1, 1>()(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(2)<T, CallType, type0, type1>(this->functionPointer);
  };
};


///////////
//// 3 ////
///////////
//! @brief ExecutorClass, that can execute Functions with three parameters.
template<class T, typename CallType, typename type0, typename type1, typename type2, template<typename, typename, int> class Evaluater = ExecutorEvaluater>
class __EXECUTOR_FUNCTIONAL_NAME(3) : public Executor<CallType>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(3) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>())
  {
    this->functionType = Executor<CallType>::FunctionDefault;
    this->functionPointer = functionPointer;
  };

  /**
   * @brief executes the Functional
   * @param object the Object the action should be executed on.
   * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType, type0, 0>()(sub, this->defaultValue),
      Evaluater<CallType, type1, 1>()(sub, this->defaultValue),
      Evaluater<CallType, type2, 2>()(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(3)<T, CallType, type0, type1, type2>(this->functionPointer);
  };
};



///////////
//// 4 ////
///////////
//! @brief ExecutorClass, that can execute Functions with four parameters.
template<class T, typename CallType, typename type0, typename type1, typename type2, typename type3, template<typename, typename, int> class Evaluater = ExecutorEvaluater>
class __EXECUTOR_FUNCTIONAL_NAME(4) : public Executor<CallType>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(4) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), ExecutorParamType<type3>())
  {
    this->functionType = Executor<CallType>::FunctionDefault;
    this->functionPointer = functionPointer;
  };

  /**
  * @brief executes the Functional
  * @param object the Object the action should be executed on.
  * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType, type0, 0>()(sub, this->defaultValue),
      Evaluater<CallType, type1, 1>()(sub, this->defaultValue),
      Evaluater<CallType, type2, 2>()(sub, this->defaultValue),
      Evaluater<CallType, type3, 3>()(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(4)<T, CallType, type0, type1, type2, type3>(this->functionPointer);
  };
};



///////////
//// 5 ////
///////////
//! @brief ExecutorClass, that can execute Functions with five parameters.
template<class T, typename CallType, typename type0, typename type1, typename type2, typename type3, typename type4, template<typename, typename, int> class Evaluater = ExecutorEvaluater>
class __EXECUTOR_FUNCTIONAL_NAME(5) : public Executor<CallType>
{
private:
  /** @brief the FunctioPointer. */
  void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3, type4) __EXECUTOR_FUNCTIONAL_CONST;

public:
  /**
   * @brief constructs the Executor.
   * @param __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER the FunctionPointer to the Calling Function.
   */
  __EXECUTOR_FUNCTIONAL_NAME(5) (void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3, type4) __EXECUTOR_FUNCTIONAL_CONST)
      : Executor<CallType>(false, ExecutorParamType<type0>(), ExecutorParamType<type1>(), ExecutorParamType<type2>(), ExecutorParamType<type3>(), ExecutorParamType<type4>())
  {
    this->functionType = Executor<CallType>::FunctionDefault;
    this->functionPointer = functionPointer;
  };

  /**
  * @brief executes the Functional
  * @param object the Object the action should be executed on.
  * @param sub the SubString to get the Parameters from.
   */
  virtual void operator()(BaseObject* object, CallType& sub = SubString()) const
  {
    (__EXECUTOR_FUNCTIONAL_FUNCTION_EXEC)(
      Evaluater<CallType, type0, 0>()(sub, this->defaultValue),
      Evaluater<CallType, type1, 1>()(sub, this->defaultValue),
      Evaluater<CallType, type2, 2>()(sub, this->defaultValue),
      Evaluater<CallType, type3, 3>()(sub, this->defaultValue),
      Evaluater<CallType, type4, 4>()(sub, this->defaultValue));
  };

  /**
   * @brief copies the Executor
   * @returns a new Executor that's a copy of this one.
   */
  virtual Executor<CallType>* clone() const
  {
    return new __EXECUTOR_FUNCTIONAL_NAME(5)<T, CallType, type0, type1, type2, type3, type4>(this->functionPointer);
  };
};









// // // // // // // // // // // // //
//// EXTENSION TO HIDE CONSTRUCT /////
// // // // // // // // // // // // //
/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 */
#define EXECUTOR_FUNCTIONAL_CREATOR0() \
template<class T> Executor<const SubString>* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)() __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(0)<T, const SubString>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR1(type0) \
template<class T> Executor<const SubString>* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(1)<T, const SubString, type0>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 * @param type1 for internal usage: the second Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR2(type0, type1) \
template<class T> Executor<const SubString>* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(2)<T, const SubString, type0, type1>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 * @param type1 for internal usage: the second Argument
 * @param type2 for internal usage: the third Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR3(type0, type1, type2) \
template<class T> Executor<const SubString>* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(3)<T, const SubString, type0, type1, type2>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 * @param type1 for internal usage: the second Argument
 * @param type2 for internal usage: the third Argument
 * @param type3 for internal usage: the fourth Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR4(type0, type1, type2, type3) \
template<class T> Executor<const SubString>* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
  return new __EXECUTOR_FUNCTIONAL_NAME(4)<T, const SubString, type0, type1, type2, type3>(functionPointer); \
}

/**
 * @brief enables us to easily retrieve an Executor of Class T with modular Argument-count, (without thinking about args at all)
 * @param type0 for internal usage: the first Argument
 * @param type1 for internal usage: the second Argument
 * @param type2 for internal usage: the third Argument
 * @param type3 for internal usage: the fourth Argument
 * @param type4 for internal usage: the fifth Argument
 */
#define EXECUTOR_FUNCTIONAL_CREATOR5(type0, type1, type2, type3, type4) \
template<class T> Executor<const SubString>* createExecutor(void (__EXECUTOR_FUNCTIONAL_FUNCTION_POINTER)(type0, type1, type2, type3, type4) __EXECUTOR_FUNCTIONAL_CONST) \
{ \
    return new __EXECUTOR_FUNCTIONAL_NAME(5)<T, const SubString, type0, type1, type2, type3, type4>(functionPointer); \
}


/**
 * Creates the FunctionCallers
 */
#define FUNCTOR_LIST(x) EXECUTOR_FUNCTIONAL_CREATOR ## x
#include "functor_list.h"
#undef FUNCTOR_LIST



#undef __EXECUTOR_FUNCTIONAL_CONST
#undef __EXECUTOR_FUNCTIONAL_NAME
#undef __EXECUTOR_FUNCTIONAL_FUNCTION_EXEC
#undef __EXECUTOR_FUNCTIONAL_FUNCTION_POINTER

#ifdef EXECUTOR_FUNCTIONAL_USE_CONST
 #undef EXECUTOR_FUNCTIONAL_USE_CONST
#endif
#ifdef EXECUTOR_FUNCTIONAL_USE_STATIC
 #undef EXECUTOR_FUNCTIONAL_USE_STATIC
#endif


#endif /* __EXECUTOR_FUNCTIONAL_NOT_INCLUDE_THIS */
#undef __EXECUTOR_FUNCTIONAL_NOT_INCLUDE_THIS
