/*!
 * @file new_class_id.h
 * @brief Definition of a dynamically allocating ClassID
 *
 */

#ifndef _NEW_CLASS_ID_H
#define _NEW_CLASS_ID_H

#include "new_object_list.h"

#include <string>
#include <list>


//! A class to dynamically allocate ClassID's and support a isA operator
class NewClassID
{
public:
  NewClassID();
  ~NewClassID();

  int leafClassID() const { return _classes.front()._objectList->id(); }

  template<class T> void registerObject(T* object, NewObjectList<T>& list);
  bool isA(const NewObjectListBase& objectList) const;
  bool isA(int classID) const;
  bool isA(const std::string& className) const;

  void listInheritance() const;

private:
  //////////////////////////////
  //// Type Definition Part ////
  //////////////////////////////
  //! A ClassEntry so we can store Classes inside of Objects
  struct ClassEntry
  {
    /** Simple Constuctor @param objectList the NewObjectList, @param iterator the (intrusive) Iterator inside of the ObjectList */
    inline ClassEntry (NewObjectListBase* objectList, NewObjectListBase::IteratorBase* iterator) : _objectList(objectList), _iterator(iterator) {}
    NewObjectListBase*                _objectList;  //!< A ObjectList this Object is part of
    NewObjectListBase::IteratorBase*  _iterator;    //!< An iterator pointing to the position of the Object inside of the List.
  };
  typedef std::list<ClassEntry>        ClassList;   //!< Type definition for the List.

  ClassList                           _classes;     //!< All Classes this object is part of.
};


/**
 * @brief Registeres an Object of Type T to objectList
 * @param object The Object to append to the objectList.
 * @param objectList The ObjectList to append the Object to.
 *
 * This function is essential to integrate objects into their designated ObjectList.
 * Remember if you do not want objects to be stored in Lists (less overhead),
 * do not attempt to call this function.
 */
template<class T>
inline void NewClassID::registerObject(T* object, NewObjectList<T>& objectList)
{
  this->_classes.push_front(ClassEntry(&objectList, objectList.registerObject(object)));
}

#endif /* _NEW_CLASS_ID_H */
