/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2006 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "new_object_list.h"
#include <cassert>


NewObjectListBase::NewObjectListBase(const std::string& className, int id)
    : _name(className)
{


  if (NewObjectListBase::_classesByID == NULL)
  {
    NewObjectListBase::_classesByID = new classIDMap;
    assert (NewObjectListBase::_classesByName == NULL);
    NewObjectListBase::_classesByName = new classNameMap;
  }
  assert(!NewObjectListBase::classNameExists(className) && "Classes should only be included once, and no two classes should have the same name (key value)");

  if (id != -1)
  {
    this->_id = id;
  }
  else
  {
    this->_id = NewObjectListBase::_classesByID->size();
    // searching for a free ID
    while (NewObjectListBase::classIDExists(_id)) ++id;
  }
  assert(!NewObjectListBase::classIDExists(id) && "Classes should only be included once, and no two classes should have the same ID (key value)");

  /// Some Output, that will fall out later
  std::cout << "register new ObjectList " << className << " ID: " << this->_id << std::endl;

  (*NewObjectListBase::_classesByID)[this->_id] = this;
  (*NewObjectListBase::_classesByName)[this->_name] = this;
}


/**
 * Destructor.
 *
 * This destructor deletes the NewObjectList, and cleans up the NewObjectList sorted Maps.
 */
NewObjectListBase::~NewObjectListBase()
{
  assert (NewObjectListBase::_classesByName != NULL && NewObjectListBase::_classesByID != NULL);

  std::cout << "Erasing: " << this->_name << " "<< this->_id  << std::endl;

  std::cout << "SIZE OF _classByID: " << NewObjectListBase::_classesByID->size() << std::endl;
  std::cout << "SIZE OF _classByName: " << NewObjectListBase::_classesByName->size() << std::endl;

  NewObjectListBase::_classesByName->erase(this->_name);
  NewObjectListBase::_classesByID->erase(this->_id);

  if (NewObjectListBase::_classesByID->empty())
  {
    delete NewObjectListBase::_classesByID;
    NewObjectListBase::_classesByID = NULL;
    assert(NewObjectListBase::_classesByName != NULL);
    delete NewObjectListBase::_classesByName;
    NewObjectListBase::_classesByName = NULL;
  }
}

NewObjectListBase::classIDMap* NewObjectListBase::_classesByID = NULL;
NewObjectListBase::classNameMap* NewObjectListBase::_classesByName = NULL;

/**
 * @returns the Registered Class Count.
 */
unsigned int NewObjectListBase::classCount()
{
  assert (NewObjectListBase::_classesByID != NULL);
  return NewObjectListBase::_classesByID->size();
};

/**
 * @brief Checks if a Class with name already exists.
 * @param name The Name of the Class to check.
 * @return true if such a class already exists.
 */
bool NewObjectListBase::classNameExists(const std::string& name)
{
  return (NewObjectListBase::_classesByName->find(name) != NewObjectListBase::_classesByName->end());
}

/**
 * @brief Checks if a Class with name already exists.
 * @param name The Name of the Class to check.
 * @return true if such a class already exists.
 */
bool NewObjectListBase::classIDExists(int id)
{
  return (NewObjectListBase::_classesByID->find(id) != NewObjectListBase::_classesByID->end());
}


/**
 * @brief Converts an ID into a ClassName String.
 * @param classID The ID to convert.
 * @return The ClassName or an empty string if the ID was not found.
 */
const std::string& NewObjectListBase::IDToString(int classID)
{
  assert (NewObjectListBase::_classesByID != NULL);
  NewObjectListBase::classIDMap::iterator it = NewObjectListBase::_classesByID->find(classID);
  if (it != NewObjectListBase::_classesByID->end())
    return (*it).second->name();
  else
  {
    static std::string empty;
    return empty;
  }
}


/**
 * @brief Converts a String into an ID
 * @param className the Name of the Class to search for
 * @return The Classes ID if found, -1 otherwise.
 */
int NewObjectListBase::StringToID(const std::string& className)
{
  assert (NewObjectListBase::_classesByName != NULL);
  NewObjectListBase::classNameMap::iterator it = NewObjectListBase::_classesByName->find(className);
  if (it != NewObjectListBase::_classesByName->end())
    return (*it).second->id();
  else
    return -1;
}

