/*!
 * @file network_stream.h
 *  implementation of a network pipe
 */

#ifndef _NETWORK_STREAM
#define _NETWORK_STREAM

#include <vector>
#include <list>
#include <map>

#include "data_stream.h"
#include "server_socket.h"
#include "handshake.h"
#include "monitor/connection_monitor.h"
#include "udp_server_socket.h"
#include "peer_info.h"

#include "shared_network_data.h"

class Synchronizeable;
class NetworkSocket;
class ServerSocket;
class NetworkGameManager;
class NetworkMonitor;


typedef std::list<Synchronizeable*>  SynchronizeableList;
typedef std::map<int,PeerInfo>       PeerList;


class NetworkStream : public DataStream
{

  public:
    NetworkStream();
    NetworkStream(int nodeType);
    virtual ~NetworkStream();

    void init();

    /* network interface controls */
    void connectToMasterServer(std::string host, int port);
    void connectToProxyServer(int proxyId, std::string host, int port);
    void createServer(int clientPort, int proxyPort, int clientSoftPort);

    void createNetworkGameManager();
    void startHandshake(int userId = NET_ID_MASTER_SERVER);

    void reconnectToServer(IP address);
    void softReconnectToServer(int serverUserId, IP address);
    void prepareSoftConnection(int userId);


    /* synchronizeable interface */
    void connectSynchronizeable(Synchronizeable& sync);
    void disconnectSynchronizeable(Synchronizeable& sync);

    inline int getMaxConnections(){ return SharedNetworkData::getInstance()->getMaxPlayer(); }

    /* functions for the peerInfo information retreival */
    /** @returns true if this userId is activated at this client false if not*/
    inline bool isUserIdActive( int userID ) { return (peers.find(userID) != peers.end()); }
    /** @returns true if this userId is a local client */
    inline bool isUserLocal( int userID) { return this->isUserIdActive(userID); }
    /** @returns true if this user is a master server */
    inline bool isUserMasterServer( int userID ){ if ( !isUserIdActive(userID) ) return false; return peers[userID].isMasterServer(); }
    /** @returns true if this user is a proxy server */
    inline bool isUserProxyServerActive( int userID ){ if ( !isUserIdActive(userID) ) return false; return peers[userID].isProxyServerActive(); }
    /** @returns true if this user is a client */
    inline bool isUserClient( int userID ){ if ( !isUserIdActive(userID) ) return false; return peers[userID].isClient(); }

    /* peering interface */
    inline PeerInfo & getPeerInfo( int userId ) { return peers[userId]; }
    inline PeerInfo* getPeerInfo() { return this->pInfo; }
    inline PeerList getPeerInfoList() { return this->peers; }

    inline void setRedirectionTest() { this->bSoftRedirect = true; }

    /* data processing*/
    virtual void processData();

    /* debugging */
    void debug();


  private:

    inline SynchronizeableList::const_iterator getSyncBegin(){ return synchronizeables.begin(); }
    inline SynchronizeableList::const_iterator getSyncEnd(){ return synchronizeables.end(); }
    void cleanUpOldSyncList();
    int getSyncCount();

    void updateConnectionList();

    /* handle processes */
    void handleHandshakes();
    void handleUpstream( int tick );
    void handleDownstream(int tick );

    /* handle events*/
    void handleNewClient( int userId );

    void handleConnect( int userId);
    void handleSoftConnect( int userId);
    void handleReconnect( int userId );
    void handleDisconnect( int userId );
    void handleSoftDisconnect( int userId);

    void writeToNewDict( byte * data, int length, bool upstream );


  private:
    PeerList                   peers;                       //!< list of the network node informations

    PeerInfo*                  pInfo;                       //!< the info about the local peer node (not in the PeerList)

    std::list<int>             freeSocketSlots;             //!< list of free sockets (to ensure not to resycle sockets)
    int                        currentState;                //!< current state id

    NetworkMonitor*            networkMonitor;              //!< the network monitor
    NetworkGameManager*        networkGameManager;          //!< reference to the network game manager
    ServerSocket*              clientSocket;                //!< the listening socket for clients of the server
    ServerSocket*              clientSoftSocket;            //!< the listening socket for soft connections to the server
    ServerSocket*              proxySocket;                 //!< socket for proxy connections

    std::map<int,int>          oldSynchronizeables;         //!< used to save recently deleted sync ids to not recreate them
    SynchronizeableList        synchronizeables;            //!< list of the synchronizeables

    byte                       buf[UDP_PACKET_SIZE];        //!< used by handleUp/Downstream
    byte                       compBuf[UDP_PACKET_SIZE];    //!< used by handleUp/Downstream
    int                        remainingBytesToWriteToDict; //!< if > 0 NetworkStream will write packets to DATA/dicts/newdict

    int                        dictServer;                  //!< the zip dict for the server
    int                        dictClient;                  //!< the zip dict for the client

    bool                       bRedirect;                   //!< true if the master server sent a redirect command
    int                        redirectionUID;              //!< uid of the redir host
    bool                       bSoftRedirect;               //!< tsting
};
#endif /* _NETWORK_STREAM */
